"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.toBeObservableComparer = exports.toHaveSubscriptionsComparer = exports.materializeInnerObservable = void 0;
var rxjs_1 = require("rxjs");
var testing_1 = require("rxjs/testing");
var lodash_1 = require("lodash");
var scheduler_1 = require("./scheduler");
var map_symbols_to_notifications_1 = require("./map-symbols-to-notifications");
var marble_unparser_1 = require("./marble-unparser");
/*
 * Based on source code found in rxjs library
 * https://github.com/ReactiveX/rxjs/blob/master/src/testing/TestScheduler.ts
 *
 */
function materializeInnerObservable(observable, outerFrame) {
    var messages = [];
    var scheduler = scheduler_1.getTestScheduler();
    observable.subscribe({
        next: function (value) {
            messages.push({
                frame: scheduler.frame - outerFrame,
                notification: {
                    kind: 'N',
                    value: value,
                    error: undefined,
                },
            });
        },
        error: function (error) {
            messages.push({
                frame: scheduler.frame - outerFrame,
                notification: {
                    kind: 'E',
                    value: undefined,
                    error: error,
                },
            });
        },
        complete: function () {
            messages.push({
                frame: scheduler.frame - outerFrame,
                notification: {
                    kind: 'C',
                    value: undefined,
                    error: undefined,
                },
            });
        },
    });
    return messages;
}
exports.materializeInnerObservable = materializeInnerObservable;
function toHaveSubscriptionsComparer(actual, marbles) {
    var marblesArray = typeof marbles === 'string' ? [marbles] : marbles;
    var results = marblesArray.map(function (marbles) {
        return testing_1.TestScheduler.parseMarblesAsSubscriptions(marbles);
    });
    expect(results).toEqual(actual.getSubscriptions());
    return { pass: true, message: function () { return ''; } };
}
exports.toHaveSubscriptionsComparer = toHaveSubscriptionsComparer;
function toBeObservableComparer(actual, fixture) {
    var results = [];
    var subscription;
    var scheduler = scheduler_1.getTestScheduler();
    scheduler.schedule(function () {
        subscription = actual.subscribe({
            next: function (x) {
                var value = x;
                // Support Observable-of-Observables
                if (x instanceof rxjs_1.Observable) {
                    value = materializeInnerObservable(value, scheduler.frame);
                }
                results.push({
                    frame: scheduler.frame,
                    notification: {
                        kind: 'N',
                        value: value,
                        error: undefined,
                    },
                });
            },
            error: function (error) {
                results.push({
                    frame: scheduler.frame,
                    notification: {
                        kind: 'E',
                        value: undefined,
                        error: error,
                    },
                });
            },
            complete: function () {
                results.push({
                    frame: scheduler.frame,
                    notification: {
                        kind: 'C',
                        value: undefined,
                        error: undefined,
                    },
                });
            },
        });
    });
    scheduler.flush();
    var expected = testing_1.TestScheduler.parseMarbles(fixture.marbles, fixture.values, fixture.error, true, true);
    try {
        expect(results).toEqual(expected);
        return { pass: true, message: function () { return ''; } };
    }
    catch (e) {
        var mapNotificationToSymbol = buildNotificationToSymbolMapper(fixture.marbles, expected, lodash_1.isEqual);
        var receivedMarble = marble_unparser_1.unparseMarble(results, mapNotificationToSymbol);
        var message_1 = formatMessage(fixture.marbles, expected, receivedMarble, results);
        return { pass: false, message: function () { return message_1; } };
    }
}
exports.toBeObservableComparer = toBeObservableComparer;
function buildNotificationToSymbolMapper(expectedMarbles, expectedMessages, equalityFn) {
    var symbolsToNotificationsMap = map_symbols_to_notifications_1.mapSymbolsToNotifications(expectedMarbles, expectedMessages);
    return function (notification) {
        var mapped = Object.keys(symbolsToNotificationsMap).find(function (key) {
            return equalityFn(symbolsToNotificationsMap[key], notification);
        });
        return mapped || '?';
    };
}
function formatMessage(expectedMarbles, expectedMessages, receivedMarbles, receivedMessages) {
    return "\n    Expected: " + expectedMarbles + ",\n    Received: " + receivedMarbles + ",\n    \n    Expected:\n    " + JSON.stringify(expectedMessages) + "\n    \n    Received:\n    " + JSON.stringify(receivedMessages) + ",\n  ";
}
//# sourceMappingURL=utils.js.map