import { base64UrlEncode } from '../base64-helper';
/**
 * Interface for Handlers that are hooked in to
 * validate tokens.
 */
export class ValidationHandler {
}
/**
 * This abstract implementation of ValidationHandler already implements
 * the method validateAtHash. However, to make use of it,
 * you have to override the method calcHash.
 */
export class AbstractValidationHandler {
    /**
     * Validates the at_hash in an id_token against the received access_token.
     */
    async validateAtHash(params) {
        let hashAlg = this.inferHashAlgorithm(params.idTokenHeader);
        let tokenHash = await this.calcHash(params.accessToken, hashAlg); // sha256(accessToken, { asString: true });
        let leftMostHalf = tokenHash.substr(0, tokenHash.length / 2);
        let atHash = base64UrlEncode(leftMostHalf);
        let claimsAtHash = params.idTokenClaims['at_hash'].replace(/=/g, '');
        if (atHash !== claimsAtHash) {
            console.error('exptected at_hash: ' + atHash);
            console.error('actual at_hash: ' + claimsAtHash);
        }
        return atHash === claimsAtHash;
    }
    /**
     * Infers the name of the hash algorithm to use
     * from the alg field of an id_token.
     *
     * @param jwtHeader the id_token's parsed header
     */
    inferHashAlgorithm(jwtHeader) {
        let alg = jwtHeader['alg'];
        if (!alg.match(/^.S[0-9]{3}$/)) {
            throw new Error('Algorithm not supported: ' + alg);
        }
        return 'sha-' + alg.substr(2);
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidmFsaWRhdGlvbi1oYW5kbGVyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vLi4vcHJvamVjdHMvbGliL3NyYy90b2tlbi12YWxpZGF0aW9uL3ZhbGlkYXRpb24taGFuZGxlci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQSxPQUFPLEVBQUUsZUFBZSxFQUFFLE1BQU0sa0JBQWtCLENBQUM7QUFXbkQ7OztHQUdHO0FBQ0gsTUFBTSxPQUFnQixpQkFBaUI7Q0FjdEM7QUFFRDs7OztHQUlHO0FBQ0gsTUFBTSxPQUFnQix5QkFBeUI7SUFNN0M7O09BRUc7SUFDSCxLQUFLLENBQUMsY0FBYyxDQUFDLE1BQXdCO1FBQzNDLElBQUksT0FBTyxHQUFHLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxNQUFNLENBQUMsYUFBYSxDQUFDLENBQUM7UUFFNUQsSUFBSSxTQUFTLEdBQUcsTUFBTSxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxXQUFXLEVBQUUsT0FBTyxDQUFDLENBQUMsQ0FBQywyQ0FBMkM7UUFFN0csSUFBSSxZQUFZLEdBQUcsU0FBUyxDQUFDLE1BQU0sQ0FBQyxDQUFDLEVBQUUsU0FBUyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQztRQUU3RCxJQUFJLE1BQU0sR0FBRyxlQUFlLENBQUMsWUFBWSxDQUFDLENBQUM7UUFFM0MsSUFBSSxZQUFZLEdBQUcsTUFBTSxDQUFDLGFBQWEsQ0FBQyxTQUFTLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRXJFLElBQUksTUFBTSxLQUFLLFlBQVksRUFBRTtZQUMzQixPQUFPLENBQUMsS0FBSyxDQUFDLHFCQUFxQixHQUFHLE1BQU0sQ0FBQyxDQUFDO1lBQzlDLE9BQU8sQ0FBQyxLQUFLLENBQUMsa0JBQWtCLEdBQUcsWUFBWSxDQUFDLENBQUM7U0FDbEQ7UUFFRCxPQUFPLE1BQU0sS0FBSyxZQUFZLENBQUM7SUFDakMsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ08sa0JBQWtCLENBQUMsU0FBaUI7UUFDNUMsSUFBSSxHQUFHLEdBQVcsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRW5DLElBQUksQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQyxFQUFFO1lBQzlCLE1BQU0sSUFBSSxLQUFLLENBQUMsMkJBQTJCLEdBQUcsR0FBRyxDQUFDLENBQUM7U0FDcEQ7UUFFRCxPQUFPLE1BQU0sR0FBRyxHQUFHLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ2hDLENBQUM7Q0FhRiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IGJhc2U2NFVybEVuY29kZSB9IGZyb20gJy4uL2Jhc2U2NC1oZWxwZXInO1xuXG5leHBvcnQgaW50ZXJmYWNlIFZhbGlkYXRpb25QYXJhbXMge1xuICBpZFRva2VuOiBzdHJpbmc7XG4gIGFjY2Vzc1Rva2VuOiBzdHJpbmc7XG4gIGlkVG9rZW5IZWFkZXI6IG9iamVjdDtcbiAgaWRUb2tlbkNsYWltczogb2JqZWN0O1xuICBqd2tzOiBvYmplY3Q7XG4gIGxvYWRLZXlzOiAoKSA9PiBQcm9taXNlPG9iamVjdD47XG59XG5cbi8qKlxuICogSW50ZXJmYWNlIGZvciBIYW5kbGVycyB0aGF0IGFyZSBob29rZWQgaW4gdG9cbiAqIHZhbGlkYXRlIHRva2Vucy5cbiAqL1xuZXhwb3J0IGFic3RyYWN0IGNsYXNzIFZhbGlkYXRpb25IYW5kbGVyIHtcbiAgLyoqXG4gICAqIFZhbGlkYXRlcyB0aGUgc2lnbmF0dXJlIG9mIGFuIGlkX3Rva2VuLlxuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IHZhbGlkYXRlU2lnbmF0dXJlKFxuICAgIHZhbGlkYXRpb25QYXJhbXM6IFZhbGlkYXRpb25QYXJhbXNcbiAgKTogUHJvbWlzZTxhbnk+O1xuXG4gIC8qKlxuICAgKiBWYWxpZGF0ZXMgdGhlIGF0X2hhc2ggaW4gYW4gaWRfdG9rZW4gYWdhaW5zdCB0aGUgcmVjZWl2ZWQgYWNjZXNzX3Rva2VuLlxuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IHZhbGlkYXRlQXRIYXNoKFxuICAgIHZhbGlkYXRpb25QYXJhbXM6IFZhbGlkYXRpb25QYXJhbXNcbiAgKTogUHJvbWlzZTxib29sZWFuPjtcbn1cblxuLyoqXG4gKiBUaGlzIGFic3RyYWN0IGltcGxlbWVudGF0aW9uIG9mIFZhbGlkYXRpb25IYW5kbGVyIGFscmVhZHkgaW1wbGVtZW50c1xuICogdGhlIG1ldGhvZCB2YWxpZGF0ZUF0SGFzaC4gSG93ZXZlciwgdG8gbWFrZSB1c2Ugb2YgaXQsXG4gKiB5b3UgaGF2ZSB0byBvdmVycmlkZSB0aGUgbWV0aG9kIGNhbGNIYXNoLlxuICovXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgQWJzdHJhY3RWYWxpZGF0aW9uSGFuZGxlciBpbXBsZW1lbnRzIFZhbGlkYXRpb25IYW5kbGVyIHtcbiAgLyoqXG4gICAqIFZhbGlkYXRlcyB0aGUgc2lnbmF0dXJlIG9mIGFuIGlkX3Rva2VuLlxuICAgKi9cbiAgYWJzdHJhY3QgdmFsaWRhdGVTaWduYXR1cmUodmFsaWRhdGlvblBhcmFtczogVmFsaWRhdGlvblBhcmFtcyk6IFByb21pc2U8YW55PjtcblxuICAvKipcbiAgICogVmFsaWRhdGVzIHRoZSBhdF9oYXNoIGluIGFuIGlkX3Rva2VuIGFnYWluc3QgdGhlIHJlY2VpdmVkIGFjY2Vzc190b2tlbi5cbiAgICovXG4gIGFzeW5jIHZhbGlkYXRlQXRIYXNoKHBhcmFtczogVmFsaWRhdGlvblBhcmFtcyk6IFByb21pc2U8Ym9vbGVhbj4ge1xuICAgIGxldCBoYXNoQWxnID0gdGhpcy5pbmZlckhhc2hBbGdvcml0aG0ocGFyYW1zLmlkVG9rZW5IZWFkZXIpO1xuXG4gICAgbGV0IHRva2VuSGFzaCA9IGF3YWl0IHRoaXMuY2FsY0hhc2gocGFyYW1zLmFjY2Vzc1Rva2VuLCBoYXNoQWxnKTsgLy8gc2hhMjU2KGFjY2Vzc1Rva2VuLCB7IGFzU3RyaW5nOiB0cnVlIH0pO1xuXG4gICAgbGV0IGxlZnRNb3N0SGFsZiA9IHRva2VuSGFzaC5zdWJzdHIoMCwgdG9rZW5IYXNoLmxlbmd0aCAvIDIpO1xuXG4gICAgbGV0IGF0SGFzaCA9IGJhc2U2NFVybEVuY29kZShsZWZ0TW9zdEhhbGYpO1xuXG4gICAgbGV0IGNsYWltc0F0SGFzaCA9IHBhcmFtcy5pZFRva2VuQ2xhaW1zWydhdF9oYXNoJ10ucmVwbGFjZSgvPS9nLCAnJyk7XG5cbiAgICBpZiAoYXRIYXNoICE9PSBjbGFpbXNBdEhhc2gpIHtcbiAgICAgIGNvbnNvbGUuZXJyb3IoJ2V4cHRlY3RlZCBhdF9oYXNoOiAnICsgYXRIYXNoKTtcbiAgICAgIGNvbnNvbGUuZXJyb3IoJ2FjdHVhbCBhdF9oYXNoOiAnICsgY2xhaW1zQXRIYXNoKTtcbiAgICB9XG5cbiAgICByZXR1cm4gYXRIYXNoID09PSBjbGFpbXNBdEhhc2g7XG4gIH1cblxuICAvKipcbiAgICogSW5mZXJzIHRoZSBuYW1lIG9mIHRoZSBoYXNoIGFsZ29yaXRobSB0byB1c2VcbiAgICogZnJvbSB0aGUgYWxnIGZpZWxkIG9mIGFuIGlkX3Rva2VuLlxuICAgKlxuICAgKiBAcGFyYW0gand0SGVhZGVyIHRoZSBpZF90b2tlbidzIHBhcnNlZCBoZWFkZXJcbiAgICovXG4gIHByb3RlY3RlZCBpbmZlckhhc2hBbGdvcml0aG0oand0SGVhZGVyOiBvYmplY3QpOiBzdHJpbmcge1xuICAgIGxldCBhbGc6IHN0cmluZyA9IGp3dEhlYWRlclsnYWxnJ107XG5cbiAgICBpZiAoIWFsZy5tYXRjaCgvXi5TWzAtOV17M30kLykpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQWxnb3JpdGhtIG5vdCBzdXBwb3J0ZWQ6ICcgKyBhbGcpO1xuICAgIH1cblxuICAgIHJldHVybiAnc2hhLScgKyBhbGcuc3Vic3RyKDIpO1xuICB9XG5cbiAgLyoqXG4gICAqIENhbGN1bGF0ZXMgdGhlIGhhc2ggZm9yIHRoZSBwYXNzZWQgdmFsdWUgYnkgdXNpbmdcbiAgICogdGhlIHBhc3NlZCBoYXNoIGFsZ29yaXRobS5cbiAgICpcbiAgICogQHBhcmFtIHZhbHVlVG9IYXNoXG4gICAqIEBwYXJhbSBhbGdvcml0aG1cbiAgICovXG4gIHByb3RlY3RlZCBhYnN0cmFjdCBjYWxjSGFzaChcbiAgICB2YWx1ZVRvSGFzaDogc3RyaW5nLFxuICAgIGFsZ29yaXRobTogc3RyaW5nXG4gICk6IFByb21pc2U8c3RyaW5nPjtcbn1cbiJdfQ==