"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.migrateDefaultsSchematic = exports.moveDirectory = exports.migrateStorybookInstance = exports.migrateAllStorybookInstances = exports.migrateDefaultsGenerator = void 0;
const devkit_1 = require("@nrwl/devkit");
const semver_1 = require("semver");
const versions_1 = require("../../utils/versions");
const path_1 = require("path");
const version_utils_1 = require("@nrwl/workspace/src/utilities/version-utils");
const typescript_1 = require("@nrwl/workspace/src/utilities/typescript");
function migrateDefaultsGenerator(tree, schema) {
    if (schema.all === undefined) {
        schema.all = true;
    }
    if (schema.keepOld === undefined) {
        schema.keepOld = true;
    }
    /**
     * The "all" flag - or the absense of "name"
     * should indicate that all Storybook instances everywhere in the
     * project should be migrated.
     *
     * Not running a migration for "all" does NOT make
     * sense, since everything links back to the root .storybook
     * directory, which will get migrated.
     * However, someone may want to do it step by step. But all should be migrated
     * eventually.
     */
    let configFolder;
    let uiFramework;
    const runAll = schema.all && !schema.name;
    if (!runAll && schema.name) {
        const projectConfig = (0, devkit_1.readProjectConfiguration)(tree, schema.name);
        if (projectConfig.targets && projectConfig.targets.storybook) {
            configFolder =
                projectConfig.targets.storybook.options.config.configFolder;
            uiFramework = projectConfig.targets.storybook.options.uiFramework;
        }
        if (projectConfig.targets && projectConfig.targets.storybook) {
            configFolder =
                projectConfig.targets.storybook.options.config.configFolder;
            uiFramework = projectConfig.targets.storybook.options.uiFramework;
        }
    }
    if (!runAll && configFolder) {
        migrateStorybookInstance(tree, schema.keepOld, schema.name, uiFramework, configFolder);
    }
    else {
        migrateAllStorybookInstances(tree, schema.keepOld);
    }
    if (runAll) {
        migrateRootLevelStorybookInstance(tree, schema.keepOld);
        return upgradeStorybookPackagesInPackageJson(tree);
    }
}
exports.migrateDefaultsGenerator = migrateDefaultsGenerator;
function migrateAllStorybookInstances(tree, keepOld) {
    devkit_1.logger.debug('adding .storybook folder to project');
    const projects = (0, devkit_1.getProjects)(tree);
    const projectsThatHaveStorybookConfiguration = [...projects.entries()]
        .filter(([_, projectConfig]) => projectConfig.targets && projectConfig.targets.storybook)
        .map(([projectName, projectConfig]) => {
        if (projectConfig.targets && projectConfig.targets.storybook) {
            return {
                name: projectName,
                uiFramework: projectConfig.targets.storybook.options.uiFramework,
                configFolder: projectConfig.targets.storybook.options.config.configFolder,
            };
        }
    });
    for (const projectWithStorybook of projectsThatHaveStorybookConfiguration) {
        migrateStorybookInstance(tree, keepOld, projectWithStorybook.name, projectWithStorybook.uiFramework, projectWithStorybook.configFolder);
    }
}
exports.migrateAllStorybookInstances = migrateAllStorybookInstances;
function migrateStorybookInstance(tree, keepOld, projectName, uiFramework, configFolder) {
    migrateProjectLevelStorybookInstance(tree, keepOld, projectName, uiFramework, configFolder);
}
exports.migrateStorybookInstance = migrateStorybookInstance;
function maybeUpdateVersion(tree) {
    let needsInstall = false;
    (0, devkit_1.updateJson)(tree, 'package.json', (json) => {
        json.dependencies = json.dependencies || {};
        json.devDependencies = json.devDependencies || {};
        const allStorybookPackagesInDependencies = Object.keys(json.dependencies).filter((packageName) => packageName.startsWith('@storybook/') &&
            !packageName.includes('@storybook/react-native'));
        const allStorybookPackagesInDevDependencies = Object.keys(json.devDependencies).filter((packageName) => packageName.startsWith('@storybook/') &&
            !packageName.includes('@storybook/react-native'));
        const storybookPackages = [
            ...allStorybookPackagesInDependencies,
            ...allStorybookPackagesInDevDependencies,
        ];
        storybookPackages.forEach((storybookPackageName) => {
            if (json.dependencies[storybookPackageName]) {
                const version = (0, version_utils_1.checkAndCleanWithSemver)(storybookPackageName, json.dependencies[storybookPackageName]);
                if ((0, semver_1.lte)(version, '6.0.0')) {
                    json.dependencies[storybookPackageName] = versions_1.storybookVersion;
                    needsInstall = true;
                }
            }
            if (json.devDependencies[storybookPackageName]) {
                const version = (0, version_utils_1.checkAndCleanWithSemver)(storybookPackageName, json.devDependencies[storybookPackageName]);
                if ((0, semver_1.lte)(version, '6.0.0')) {
                    json.devDependencies[storybookPackageName] = versions_1.storybookVersion;
                    needsInstall = true;
                }
            }
        });
        return json;
    });
    if (needsInstall) {
        return () => (0, devkit_1.installPackagesTask)(tree);
    }
}
function upgradeStorybookPackagesInPackageJson(tree) {
    /**
     * Should upgrade all @storybook/* packages in package.json
     */
    return maybeUpdateVersion(tree);
}
function deleteOldFiles(tree, configFolderDir) {
    (0, devkit_1.visitNotIgnoredFiles)(tree, configFolderDir, (file) => {
        if (file.includes('addons.js') || file.includes('config.js')) {
            tree.delete(file);
        }
    });
    // tree.delete(configFolderDir);
}
function moveOldFiles(tree, configFolderDir) {
    moveDirectory(tree, configFolderDir, configFolderDir.replace('.storybook', '.old_storybook'));
    // tree.delete(configFolderDir);
}
function migrateProjectLevelStorybookInstance(tree, keepOld, projectName, uiFramework, configFolder) {
    const { root, projectType } = (0, devkit_1.readProjectConfiguration)(tree, projectName);
    const projectDirectory = projectType === 'application' ? 'app' : 'lib';
    const old_folder_exists_already = tree.exists(configFolder.replace('.storybook', '.old_storybook'));
    const new_config_exists_already = tree.exists(`${configFolder}/main.js`);
    if (old_folder_exists_already || new_config_exists_already) {
        return;
    }
    if (keepOld) {
        moveOldFiles(tree, configFolder);
    }
    else {
        deleteOldFiles(tree, configFolder);
    }
    if (tree.exists(configFolder)) {
        return;
    }
    (0, devkit_1.generateFiles)(tree, (0, path_1.join)(__dirname, '../configuration/project-files/.storybook'), configFolder, {
        tmpl: '',
        uiFramework,
        offsetFromRoot: (0, devkit_1.offsetFromRoot)(root),
        rootTsConfigPath: (0, typescript_1.getRootTsConfigPathInTree)(tree),
        projectType: projectDirectory,
        useWebpack5: uiFramework === '@storybook/angular',
        existsRootWebpackConfig: tree.exists('.storybook/webpack.config.js'),
    });
}
function migrateRootLevelStorybookInstance(tree, keepOld) {
    const old_folder_exists_already = tree.exists('.old_storybook');
    const new_config_exists_already = tree.exists(`.storybook/main.js`);
    if (old_folder_exists_already || new_config_exists_already) {
        return;
    }
    if (keepOld) {
        moveOldFiles(tree, '.storybook');
    }
    else {
        deleteOldFiles(tree, '.storybook');
    }
    (0, devkit_1.generateFiles)(tree, (0, path_1.join)(__dirname, '../configuration/root-files/.storybook'), '.storybook', { rootTsConfigPath: (0, typescript_1.getRootTsConfigPathInTree)(tree) });
}
function moveDirectory(tree, from, to) {
    (0, devkit_1.visitNotIgnoredFiles)(tree, from, (file) => {
        tree.rename(file, file.replace(from, to));
    });
}
exports.moveDirectory = moveDirectory;
exports.default = migrateDefaultsGenerator;
exports.migrateDefaultsSchematic = (0, devkit_1.convertNxGenerator)(migrateDefaultsGenerator);
//# sourceMappingURL=migrate-defaults-5-to-6.js.map