"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const devkit_1 = require("@nrwl/devkit");
const tsquery_1 = require("@phenomnomnominal/tsquery");
const path_1 = require("path");
const ts = require("typescript");
function default_1(tree) {
    var _a, _b, _c;
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const projects = (0, devkit_1.getProjects)(tree);
        for (const [projectName, project] of projects.entries()) {
            // only interested in projects with @nrwl/jest:jest executor
            if (((_b = (_a = project.targets) === null || _a === void 0 ? void 0 : _a.test) === null || _b === void 0 ? void 0 : _b.executor) !== '@nrwl/jest:jest') {
                continue;
            }
            const jestConfigPath = (_c = project.targets.test.options) === null || _c === void 0 ? void 0 : _c.jestConfig;
            // if there's not jestConfigPath, we don't process it
            if (!jestConfigPath) {
                devkit_1.logger.warn(`The "jestConfig" property is not configured for the test target of the project "${projectName}". Skipping it.`);
                continue;
            }
            if (!tree.exists(jestConfigPath)) {
                devkit_1.logger.warn(`The specified "jestConfig" path "${jestConfigPath}" for the project "${projectName}" can not be found. Skipping it.`);
                continue;
            }
            let jestConfigContents = tree.read(jestConfigPath, 'utf-8');
            // check if it's an angular project by looking for jest-preset-angular
            if (!jestConfigContents.includes('jest-preset-angular')) {
                continue;
            }
            const { shouldUpdate, testSetupFilePath } = shouldUpdateTeardownConfig(tree, projectName, jestConfigPath, jestConfigContents);
            if (!shouldUpdate) {
                continue;
            }
            const printer = ts.createPrinter();
            if (shouldUpdate === 'testSetup') {
                optOutTestTeardownFromTestSetupFile(tree, testSetupFilePath, printer);
            }
            else {
                optOutTestTeardownFromTestFiles(tree, projectName, printer);
            }
        }
        yield (0, devkit_1.formatFiles)(tree);
    });
}
exports.default = default_1;
function shouldUpdateTeardownConfig(tree, projectName, jestConfigPath, jestConfigContents) {
    const jestConfigAst = tsquery_1.tsquery.ast(jestConfigContents);
    const setupFilesAfterEnvSelector = 'PropertyAssignment:has(Identifier[name=setupFilesAfterEnv])';
    const setupFilesAfterEnvProperty = (0, tsquery_1.tsquery)(jestConfigAst, setupFilesAfterEnvSelector, {
        visitAllChildren: true,
    })[0];
    // no property specified, we try to migrate test files
    if (!setupFilesAfterEnvProperty) {
        return { shouldUpdate: 'testFiles' };
    }
    const setupFilesValueSelector = 'ArrayLiteralExpression StringLiteral';
    const setupFilesValue = (0, tsquery_1.tsquery)(setupFilesAfterEnvProperty, setupFilesValueSelector, {
        visitAllChildren: true,
    })[0];
    // not setup file specified, we try to migrate test files
    if (!setupFilesValue) {
        return { shouldUpdate: 'testFiles' };
    }
    const testSetupFilePath = setupFilesValue
        .getText()
        .replace('<rootDir>', (0, path_1.dirname)(jestConfigPath))
        .replace(/['"]/g, '');
    // the specified file is invalid, we migrate test files
    if (!tree.exists(testSetupFilePath)) {
        return { shouldUpdate: 'testFiles' };
    }
    let testSetupFileContents = tree.read(testSetupFilePath, 'utf-8');
    const testSetupFileAst = tsquery_1.tsquery.ast(testSetupFileContents);
    const initTestEnvironmentSelector = 'CallExpression:has(PropertyAccessExpression:has(Identifier[name=initTestEnvironment]))';
    const initTestEnvironmentCall = (0, tsquery_1.tsquery)(testSetupFileAst, initTestEnvironmentSelector, {
        visitAllChildren: true,
    })[0];
    // no initTestEnvironment call, we migrate the test setup file
    if (!initTestEnvironmentCall) {
        return { shouldUpdate: 'testSetup', testSetupFilePath };
    }
    // no third arg, we migrate the test setup file
    if (initTestEnvironmentCall.arguments.length === 2) {
        return { shouldUpdate: 'testSetup', testSetupFilePath };
    }
    // this would be a type error, we migrate test files
    if (initTestEnvironmentCall.arguments.length < 2) {
        return { shouldUpdate: 'testFiles' };
    }
    const optionsArg = initTestEnvironmentCall.arguments[2];
    // the options arg is an object that has a teardown property, no migration is needed
    if (isObjectLiteralWithTeardown(optionsArg)) {
        return { shouldUpdate: false };
    }
    // the options arg is an object that doesn't have a teardown property, we migrate the test setup file
    if (isObjectLiteralWithoutTeardown(optionsArg)) {
        return { shouldUpdate: 'testSetup', testSetupFilePath };
    }
    // the options arg is an `aotSummaries` function, we migrate the test setup file
    if (isFunction(optionsArg)) {
        return { shouldUpdate: 'testSetup', testSetupFilePath };
    }
    // we fallback to migrate the test files
    return { shouldUpdate: 'testFiles' };
}
function optOutTestTeardownFromTestFiles(tree, projectName, printer) {
    const { sourceRoot, root } = (0, devkit_1.readProjectConfiguration)(tree, projectName);
    (0, devkit_1.visitNotIgnoredFiles)(tree, sourceRoot !== null && sourceRoot !== void 0 ? sourceRoot : root, (path) => {
        if (!['.ts', '.js'].includes((0, path_1.extname)(path))) {
            return;
        }
        let fileContents = tree.read(path, 'utf-8');
        const ast = tsquery_1.tsquery.ast(fileContents);
        const configureTestingModulePropertyAccessExpressionSelector = 'CallExpression > PropertyAccessExpression:has(Identifier[name=configureTestingModule])';
        const configureTestingModulePropertyAccessExpressions = (0, tsquery_1.tsquery)(ast, configureTestingModulePropertyAccessExpressionSelector, {
            visitAllChildren: true,
        });
        // no calls to configureTestingModule, we skip it
        if (!configureTestingModulePropertyAccessExpressions.length) {
            return;
        }
        // reverse the order to not mess with positions as we update the AST
        const reversedConfigureTestingModulePropertyAccessExpressions = sortInReverseSourceOrder(configureTestingModulePropertyAccessExpressions);
        reversedConfigureTestingModulePropertyAccessExpressions.forEach((propertyAccessExpression) => {
            const configureTestingModuleCall = propertyAccessExpression.parent;
            if (configureTestingModuleCall.arguments.length === 0 ||
                !isObjectLiteralWithoutTeardown(configureTestingModuleCall.arguments[0])) {
                return;
            }
            const testModuleMetadata = configureTestingModuleCall.arguments[0];
            const replacement = getTestModuleMetadataLiteralReplacement(testModuleMetadata, printer);
            fileContents = `${fileContents.slice(0, testModuleMetadata.getStart())}${replacement}${fileContents.slice(testModuleMetadata.getEnd())}`;
            tree.write(path, fileContents);
        });
    });
}
function optOutTestTeardownFromTestSetupFile(tree, testSetupFilePath, printer) {
    let testSetupFileContents = tree.read(testSetupFilePath, 'utf-8');
    const testSetupFileAst = tsquery_1.tsquery.ast(testSetupFileContents);
    const initTestEnvironmentSelector = 'CallExpression:has(PropertyAccessExpression:has(Identifier[name=initTestEnvironment]))';
    const initTestEnvironmentCall = (0, tsquery_1.tsquery)(testSetupFileAst, initTestEnvironmentSelector, {
        visitAllChildren: true,
    })[0];
    if (!initTestEnvironmentCall) {
        testSetupFileContents += `
      import { getTestBed } from '@angular/core/testing';
      import { BrowserDynamicTestingModule, platformBrowserDynamicTesting } from '@angular/platform-browser-dynamic/testing';

      getTestBed().resetTestEnvironment();
      getTestBed().initTestEnvironment(
        BrowserDynamicTestingModule,
        platformBrowserDynamicTesting(),
        { teardown: { destroyAfterEach: false } },
      );
    `;
        tree.write(testSetupFilePath, testSetupFileContents);
        return;
    }
    const { span, text } = getInitTestEnvironmentLiteralReplacement(initTestEnvironmentCall, printer);
    testSetupFileContents = `${testSetupFileContents.slice(0, span.start)}${text}${testSetupFileContents.slice(span.end)}`;
    tree.write(testSetupFilePath, testSetupFileContents);
}
function getInitTestEnvironmentLiteralReplacement(initTestEnvironmentCall, printer) {
    const literalProperties = [];
    const options = initTestEnvironmentCall.arguments[initTestEnvironmentCall.arguments.length - 1];
    let span;
    let prefix;
    // update the last argument of the initTestEnvironment call
    if (initTestEnvironmentCall.arguments.length === 3) {
        if (isFunction(options)) {
            // If the last argument is a function, add the function as the `aotSummaries` property.
            literalProperties.push(ts.factory.createPropertyAssignment('aotSummaries', options));
        }
        else if (ts.isObjectLiteralExpression(options)) {
            // If the property is an object literal, copy over all the properties.
            literalProperties.push(...options.properties);
        }
        prefix = '';
        span = {
            start: options.getStart(),
            end: options.getEnd(),
        };
    }
    else {
        const start = options.getEnd();
        prefix = ', ';
        span = { start, end: start };
    }
    // finally push the teardown object so that it appears last
    literalProperties.push(createTeardownAssignment());
    return {
        span,
        text: `${prefix}${printer.printNode(ts.EmitHint.Unspecified, ts.factory.createObjectLiteralExpression(literalProperties), initTestEnvironmentCall.getSourceFile())}`,
    };
}
function getTestModuleMetadataLiteralReplacement(testModuleMetadataObjectLiteral, printer) {
    return printer.printNode(ts.EmitHint.Unspecified, ts.factory.createObjectLiteralExpression([
        ...testModuleMetadataObjectLiteral.properties,
        createTeardownAssignment(),
    ], testModuleMetadataObjectLiteral.properties.length > 0), testModuleMetadataObjectLiteral.getSourceFile());
}
function isObjectLiteralWithTeardown(node) {
    return (ts.isObjectLiteralExpression(node) &&
        node.properties.some((prop) => {
            var _a;
            return ((_a = prop.name) === null || _a === void 0 ? void 0 : _a.getText()) === 'teardown';
        }));
}
function isObjectLiteralWithoutTeardown(node) {
    return (ts.isObjectLiteralExpression(node) &&
        !node.properties.find((prop) => {
            var _a;
            return ((_a = prop.name) === null || _a === void 0 ? void 0 : _a.getText()) === 'teardown';
        }));
}
function isFunction(node) {
    return (ts.isArrowFunction(node) ||
        ts.isFunctionExpression(node) ||
        ts.isFunctionDeclaration(node));
}
function createTeardownAssignment() {
    return ts.factory.createPropertyAssignment('teardown', ts.factory.createObjectLiteralExpression([
        ts.factory.createPropertyAssignment('destroyAfterEach', ts.factory.createFalse()),
    ]));
}
function sortInReverseSourceOrder(nodes) {
    return nodes.sort((a, b) => b.getEnd() - a.getEnd());
}
//# sourceMappingURL=opt-out-testbed-teardown.js.map