"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.insertNgModuleImport = exports.insertNgModuleProperty = void 0;
const devkit_1 = require("@nrwl/devkit");
const typescript_1 = require("typescript");
function insertNgModuleProperty(tree, modulePath, name, property) {
    const contents = tree.read(modulePath).toString('utf-8');
    const sourceFile = (0, typescript_1.createSourceFile)(modulePath, contents, typescript_1.ScriptTarget.ESNext);
    const coreImport = findImport(sourceFile, '@angular/core');
    if (!coreImport) {
        throw new Error(`There are no imports from "@angular/core" in ${modulePath}.`);
    }
    const ngModuleNamedImport = getNamedImport(coreImport, 'NgModule');
    const ngModuleName = ngModuleNamedImport.name.escapedText;
    const ngModuleClassDeclaration = findDecoratedClass(sourceFile, ngModuleName);
    const ngModuleDecorator = ngModuleClassDeclaration.decorators.find((decorator) => (0, typescript_1.isCallExpression)(decorator.expression) &&
        (0, typescript_1.isIdentifier)(decorator.expression.expression) &&
        decorator.expression.expression.escapedText === ngModuleName);
    const ngModuleCall = ngModuleDecorator.expression;
    if (ngModuleCall.arguments.length < 1) {
        const newContents = (0, devkit_1.applyChangesToString)(contents, [
            {
                type: devkit_1.ChangeType.Insert,
                index: ngModuleCall.getEnd() - 1,
                text: `{ ${property}: [${name}]}`,
            },
        ]);
        tree.write(modulePath, newContents);
    }
    else {
        if (!(0, typescript_1.isObjectLiteralExpression)(ngModuleCall.arguments[0])) {
            throw new Error(`The NgModule options for ${ngModuleClassDeclaration.name.escapedText} in ${modulePath} is not an object literal`);
        }
        const ngModuleOptions = ngModuleCall
            .arguments[0];
        const typeProperty = findPropertyAssignment(ngModuleOptions, property);
        if (!typeProperty) {
            let text = `${property}: [${name}]`;
            if (ngModuleOptions.properties.hasTrailingComma) {
                text = `${text},`;
            }
            else {
                text = `, ${text}`;
            }
            const newContents = (0, devkit_1.applyChangesToString)(contents, [
                {
                    type: devkit_1.ChangeType.Insert,
                    index: ngModuleOptions.getEnd() - 1,
                    text,
                },
            ]);
            tree.write(modulePath, newContents);
        }
        else {
            if (!(0, typescript_1.isArrayLiteralExpression)(typeProperty.initializer)) {
                throw new Error(`The NgModule ${property} for ${ngModuleClassDeclaration.name.escapedText} in ${modulePath} is not an array literal`);
            }
            let text;
            if (typeProperty.initializer.elements.hasTrailingComma) {
                text = `${name},`;
            }
            else {
                text = `, ${name}`;
            }
            const newContents = (0, devkit_1.applyChangesToString)(contents, [
                {
                    type: devkit_1.ChangeType.Insert,
                    index: typeProperty.initializer.getEnd() - 1,
                    text,
                },
            ]);
            tree.write(modulePath, newContents);
        }
    }
}
exports.insertNgModuleProperty = insertNgModuleProperty;
function insertNgModuleImport(tree, modulePath, importName) {
    insertNgModuleProperty(tree, modulePath, importName, 'imports');
}
exports.insertNgModuleImport = insertNgModuleImport;
function findImport(sourceFile, importPath) {
    const importStatements = sourceFile.statements.filter(typescript_1.isImportDeclaration);
    return importStatements.find((statement) => statement.moduleSpecifier
        .getText(sourceFile)
        .replace(/['"`]/g, '')
        .trim() === importPath);
}
function getNamedImport(coreImport, importName) {
    if (!(0, typescript_1.isNamedImports)(coreImport.importClause.namedBindings)) {
        throw new Error(`The import from ${coreImport.moduleSpecifier} does not have named imports.`);
    }
    return coreImport.importClause.namedBindings.elements.find((namedImport) => namedImport.propertyName
        ? (0, typescript_1.isIdentifier)(namedImport.propertyName) &&
            namedImport.propertyName.escapedText === importName
        : (0, typescript_1.isIdentifier)(namedImport.name) &&
            namedImport.name.escapedText === importName);
}
function findDecoratedClass(sourceFile, ngModuleName) {
    const classDeclarations = sourceFile.statements.filter(typescript_1.isClassDeclaration);
    return classDeclarations.find((declaration) => declaration.decorators &&
        declaration.decorators.some((decorator) => (0, typescript_1.isCallExpression)(decorator.expression) &&
            (0, typescript_1.isIdentifier)(decorator.expression.expression) &&
            decorator.expression.expression.escapedText === ngModuleName));
}
function findPropertyAssignment(ngModuleOptions, propertyName) {
    return ngModuleOptions.properties.find((property) => (0, typescript_1.isPropertyAssignment)(property) &&
        (0, typescript_1.isIdentifier)(property.name) &&
        property.name.escapedText === propertyName);
}
//# sourceMappingURL=insert-ngmodule-import.js.map