"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getModuleFilePaths = exports.getModuleDeclaredComponents = void 0;
const devkit_1 = require("@nrwl/devkit");
const typescript_1 = require("@nrwl/workspace/src/utilities/typescript");
const tsquery_1 = require("@phenomnomnominal/tsquery");
const path_1 = require("path");
const typescript_2 = require("typescript");
const ast_utils_1 = require("../../../utils/nx-devkit/ast-utils");
function getModuleDeclaredComponents(file, moduleFilePath, projectName) {
    const ngModuleDecorator = getNgModuleDecorator(file, moduleFilePath);
    const declarationsPropertyAssignment = getNgModuleDeclarationsPropertyAssignment(ngModuleDecorator, moduleFilePath, projectName);
    if (!declarationsPropertyAssignment) {
        return [];
    }
    const declarationsArray = getDeclarationsArray(file, declarationsPropertyAssignment, moduleFilePath, projectName);
    if (!declarationsArray) {
        return [];
    }
    return getDeclaredComponentsInDeclarations(declarationsArray);
}
exports.getModuleDeclaredComponents = getModuleDeclaredComponents;
function getModuleFilePaths(tree, projectPath) {
    let moduleFilePaths = [];
    (0, devkit_1.visitNotIgnoredFiles)(tree, projectPath, (filePath) => {
        if ((0, path_1.extname)(filePath) === '.ts' && hasNgModule(tree, filePath)) {
            moduleFilePaths.push(filePath);
        }
    });
    return moduleFilePaths;
}
exports.getModuleFilePaths = getModuleFilePaths;
function hasNgModule(tree, filePath) {
    const fileContent = tree.read(filePath, 'utf-8');
    const ast = tsquery_1.tsquery.ast(fileContent);
    const ngModule = (0, tsquery_1.tsquery)(ast, 'ClassDeclaration > Decorator > CallExpression > Identifier[name=NgModule]', { visitAllChildren: true });
    return ngModule.length > 0;
}
function getDeclaredComponentsInDeclarations(declarationsArray) {
    return getDeclaredComponentNodes(declarationsArray)
        .map((node) => node.getText())
        .filter((name) => name.endsWith('Component'));
}
function getDeclaredComponentNodes(declarationsArray) {
    const cmps = declarationsArray
        .getChildren()
        .find((node) => node.kind === typescript_2.SyntaxKind.SyntaxList)
        .getChildren()
        .map((node) => {
        if (node.kind === typescript_2.SyntaxKind.Identifier) {
            return node;
        }
        // if the node is a destructuring, follow the variable
        if (node.kind === typescript_2.SyntaxKind.SpreadElement) {
            const declarationVariableNode = node
                .getChildren()
                .find((node) => node.kind === typescript_2.SyntaxKind.Identifier);
            // try to find the variable declaration in the same component
            const declarationVariable = getVariableDeclaration(declarationVariableNode.getText(), declarationVariableNode.getSourceFile());
            if (declarationVariable &&
                declarationVariable.initializer.kind ===
                    typescript_2.SyntaxKind.ArrayLiteralExpression) {
                const nodes = getDeclaredComponentNodes(declarationVariable.initializer);
                return nodes;
            }
        }
        return null;
    })
        .filter((node) => !!node);
    return flatten(cmps);
}
function flatten(arr) {
    let flattened = [];
    for (const entry of arr) {
        if (Array.isArray(entry)) {
            flattened.push(...flatten(entry));
        }
        else {
            flattened.push(entry);
        }
    }
    return flattened;
}
function getDeclarationsArray(file, declarationsPropertyAssignment, moduleFilePath, projectName) {
    let declarationArray = declarationsPropertyAssignment
        .getChildren()
        .find((node) => node.kind === typescript_2.SyntaxKind.ArrayLiteralExpression);
    if (declarationArray) {
        return declarationArray;
    }
    // Attempt to follow a variable instead of the literal
    declarationArray = getModuleDeclaredComponentsFromVariable(file, declarationsPropertyAssignment);
    if (declarationArray) {
        return declarationArray;
    }
    // Attempt to follow a class declaration instead of the literal
    declarationArray = getModuleDeclaredComponentsFromClass(file, declarationsPropertyAssignment);
    if (!declarationArray) {
        devkit_1.logger.warn((0, devkit_1.stripIndents) `No stories generated because the declarations in ${moduleFilePath} is not an array literal or the variable could not be found. Hint: you can always generate stories later with the 'nx generate @nrwl/angular:stories --name=${projectName}' command.`);
    }
    return declarationArray;
}
/**
 * Try to get declared components like `declarations: someComponentsArrayConst`
 */
function getModuleDeclaredComponentsFromVariable(file, declarationsPropertyAssignment) {
    let declarationsVariable = declarationsPropertyAssignment
        .getChildren()
        .filter((node) => node.kind === typescript_2.SyntaxKind.Identifier)[1];
    if (!declarationsVariable) {
        return undefined;
    }
    // Attempt to find variable declaration in the file
    let variableDeclaration = getVariableDeclaration(declarationsVariable.getText(), file);
    if (!variableDeclaration) {
        return undefined;
    }
    const declarationArray = variableDeclaration
        .getChildren()
        .find((node) => node.kind === typescript_2.SyntaxKind.ArrayLiteralExpression);
    return declarationArray;
}
/**
 * Try to get declared components like `declarations: SomeClass.components` as in
 * https://github.com/nrwl/nx/issues/7276.
 */
function getModuleDeclaredComponentsFromClass(file, declarationsPropertyAssignment) {
    const propertyAccessExpression = declarationsPropertyAssignment
        .getChildren()
        .filter((node) => node.kind === typescript_2.SyntaxKind.PropertyAccessExpression)[0];
    if (!propertyAccessExpression) {
        return undefined;
    }
    // Should contain 2 identifiers [SomeClass, components]
    const [clazz, componentsProperty] = propertyAccessExpression
        .getChildren()
        .filter((node) => node.kind === typescript_2.SyntaxKind.Identifier);
    if (!clazz || !componentsProperty) {
        return undefined;
    }
    // Attempt to find class declaration in the file
    let classDeclaration = getClassDeclaration(clazz.getText(), file);
    if (!classDeclaration) {
        return undefined;
    }
    const declarationArray = classDeclaration.members
        .filter((node) => node.kind === typescript_2.SyntaxKind.PropertyDeclaration)
        .find((propertyDeclaration) => propertyDeclaration
        .getChildren()
        .find((node) => node.kind === typescript_2.SyntaxKind.Identifier &&
        node.getText() === componentsProperty.getText()))
        .getChildren()
        .find((node) => node.kind === typescript_2.SyntaxKind.ArrayLiteralExpression);
    return declarationArray;
}
function getClassDeclaration(className, file) {
    const classDeclaration = (0, typescript_1.findNodes)(file, typescript_2.SyntaxKind.ClassDeclaration).find((classDeclaration) => classDeclaration
        .getChildren()
        .find((node) => node.kind === typescript_2.SyntaxKind.Identifier && node.getText() === className));
    return classDeclaration;
}
function getVariableDeclaration(variableName, file) {
    const variableDeclaration = (0, typescript_1.findNodes)(file, typescript_2.SyntaxKind.VariableDeclaration).find((variableDeclaration) => variableDeclaration
        .getChildren()
        .find((node) => node.kind === typescript_2.SyntaxKind.Identifier && node.getText() === variableName));
    return variableDeclaration;
}
function getNgModuleDeclarationsPropertyAssignment(ngModuleDecorator, moduleFilePath, projectName) {
    const syntaxList = ngModuleDecorator.getChildren().find((node) => {
        return node.kind === typescript_2.SyntaxKind.SyntaxList;
    });
    const declarationsPropertyAssignment = syntaxList
        .getChildren()
        .find((node) => {
        return (node.kind === typescript_2.SyntaxKind.PropertyAssignment &&
            node.getChildren()[0].getText() === 'declarations');
    });
    if (!declarationsPropertyAssignment) {
        devkit_1.logger.warn((0, devkit_1.stripIndents) `No stories generated because there were no components declared in ${moduleFilePath}. Hint: you can always generate stories later with the 'nx generate @nrwl/angular:stories --name=${projectName}' command.`);
    }
    return declarationsPropertyAssignment;
}
function getNgModuleDecorator(file, moduleFilePath) {
    const ngModuleDecorators = (0, ast_utils_1.getDecoratorMetadata)(file, 'NgModule', '@angular/core');
    if (ngModuleDecorators.length === 0) {
        throw new Error(`No @NgModule decorator in ${moduleFilePath}.`);
    }
    return ngModuleDecorators[0];
}
//# sourceMappingURL=module-info.js.map