"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getComponentsInfo = void 0;
const devkit_1 = require("@nrwl/devkit");
const tsquery_1 = require("@phenomnomnominal/tsquery");
const path_1 = require("path");
const typescript_1 = require("typescript");
const ast_utils_1 = require("../../../utils/nx-devkit/ast-utils");
const module_info_1 = require("./module-info");
const tree_utilities_1 = require("./tree-utilities");
function getComponentsInfo(tree, moduleFilePaths, projectName) {
    return moduleFilePaths.flatMap((moduleFilePath) => {
        const file = (0, ast_utils_1.getTsSourceFile)(tree, moduleFilePath);
        const declaredComponents = (0, module_info_1.getModuleDeclaredComponents)(file, moduleFilePath, projectName);
        if (declaredComponents.length === 0) {
            return undefined;
        }
        const imports = file.statements.filter((statement) => statement.kind === typescript_1.SyntaxKind.ImportDeclaration);
        const componentsInfo = declaredComponents.map((componentName) => getComponentInfo(tree, file, imports, moduleFilePath, componentName));
        return componentsInfo;
    });
}
exports.getComponentsInfo = getComponentsInfo;
function getComponentImportPath(componentName, imports) {
    const componentImportStatement = imports.find((statement) => {
        const namedImports = statement
            .getChildren()
            .find((node) => node.kind === typescript_1.SyntaxKind.ImportClause)
            .getChildren()
            .find((node) => node.kind === typescript_1.SyntaxKind.NamedImports);
        if (namedImports === undefined)
            return false;
        const importedIdentifiers = namedImports
            .getChildren()
            .find((node) => node.kind === typescript_1.SyntaxKind.SyntaxList)
            .getChildren()
            .filter((node) => node.kind === typescript_1.SyntaxKind.ImportSpecifier)
            .map((node) => node.getText());
        return importedIdentifiers.includes(componentName);
    });
    const importPath = componentImportStatement
        .getChildren()
        .find((node) => node.kind === typescript_1.SyntaxKind.StringLiteral)
        .getText()
        .slice(1, -1);
    return importPath;
}
function getComponentInfo(tree, sourceFile, imports, moduleFilePath, componentName) {
    try {
        const moduleFolderPath = (0, path_1.dirname)(moduleFilePath);
        // try to get the component from the same file (inline scam)
        const node = (0, tsquery_1.tsquery)(sourceFile, `ClassDeclaration:has(Decorator > CallExpression > Identifier[name=Component]):has(Identifier[name=${componentName}])`, { visitAllChildren: true })[0];
        if (node) {
            return {
                componentFileName: (0, path_1.basename)(moduleFilePath, '.ts'),
                moduleFolderPath,
                name: componentName,
                path: '.',
            };
        }
        // try to get the component from the imports
        const componentFilePathRelativeToModule = getComponentImportPath(componentName, imports);
        const componentImportPath = getFullComponentFilePath(moduleFolderPath, componentFilePathRelativeToModule);
        if (tree.exists(componentImportPath) && !tree.isFile(componentImportPath)) {
            return getComponentInfoFromDir(tree, componentImportPath, componentName, moduleFolderPath);
        }
        const path = (0, path_1.dirname)(componentFilePathRelativeToModule);
        const componentFileName = (0, path_1.basename)(componentFilePathRelativeToModule);
        return { componentFileName, moduleFolderPath, name: componentName, path };
    }
    catch (ex) {
        devkit_1.logger.warn(`Could not generate a story for ${componentName}. Error: ${ex}`);
        return undefined;
    }
}
function getComponentInfoFromDir(tree, dir, componentName, moduleFolderPath) {
    let path = null;
    let componentFileName = null;
    const componentImportPathChildren = (0, tree_utilities_1.getAllFilesRecursivelyFromDir)(tree, dir);
    for (const candidateFile of componentImportPathChildren) {
        if (candidateFile.endsWith('.ts')) {
            const content = tree.read(candidateFile, 'utf-8');
            const classAndComponentRegex = new RegExp(`@Component[\\s\\S\n]*?\\bclass ${componentName}\\b`, 'g');
            if (content.match(classAndComponentRegex)) {
                path = candidateFile
                    .slice(0, candidateFile.lastIndexOf('/'))
                    .replace(moduleFolderPath, '.');
                componentFileName = candidateFile.slice(candidateFile.lastIndexOf('/') + 1, candidateFile.lastIndexOf('.'));
                break;
            }
        }
    }
    if (path === null) {
        throw new Error(`Path to component ${componentName} couldn't be found. Please open an issue on https://github.com/nrwl/nx/issues.`);
    }
    return { componentFileName, moduleFolderPath, name: componentName, path };
}
function getFullComponentFilePath(moduleFolderPath, componentFilePath) {
    if (moduleFolderPath.startsWith('/')) {
        moduleFolderPath = moduleFolderPath.slice(1, moduleFolderPath.length);
    }
    return (0, devkit_1.joinPathFragments)(moduleFolderPath, componentFilePath);
}
//# sourceMappingURL=component-info.js.map