"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createRootKarmaConfig = exports.createWorkspaceFiles = exports.updateTsLint = exports.updatePackageJson = exports.updateRootTsConfig = exports.updateWorkspaceConfigDefaults = exports.decorateAngularCli = exports.createNxJson = exports.validateWorkspace = void 0;
const tslib_1 = require("tslib");
const devkit_1 = require("@nrwl/devkit");
const workspace_1 = require("@nrwl/workspace/src/generators/workspace/workspace");
const default_base_1 = require("@nrwl/workspace/src/utilities/default-base");
const prettier_1 = require("@nrwl/workspace/src/utilities/prettier");
const typescript_1 = require("@nrwl/workspace/src/utilities/typescript");
const versions_1 = require("@nrwl/workspace/src/utils/versions");
const fs_1 = require("fs");
const path_1 = require("path");
const versions_2 = require("../../../utils/versions");
const e2e_utils_1 = require("./e2e-utils");
// TODO: most of the validation here will be moved to the app migrator when
// support for multiple apps is added. This will only contain workspace-wide
// validation.
function validateWorkspace(tree, projects) {
    try {
        if (!tree.exists('package.json')) {
            throw new Error('Cannot find package.json');
        }
        if (!tree.exists('angular.json')) {
            throw new Error('Cannot find angular.json');
        }
        // TODO: this restrictions will be removed when support for multiple
        // projects is added
        if (projects.apps.length > 2 || projects.libs.length > 0) {
            throw new Error('Can only convert projects with one app');
        }
        const e2eKey = (0, e2e_utils_1.getE2eKey)(projects);
        const e2eApp = (0, e2e_utils_1.getE2eProject)(projects);
        if (!e2eApp) {
            return;
        }
        if ((0, e2e_utils_1.isProtractorE2eProject)(e2eApp.config)) {
            if (tree.exists(e2eApp.config.targets.e2e.options.protractorConfig)) {
                return;
            }
            console.info(`Make sure the "${e2eKey}.architect.e2e.options.protractorConfig" is valid or the "${e2eKey}" project is removed from "angular.json".`);
            throw new Error(`An e2e project with Protractor was found but "${e2eApp.config.targets.e2e.options.protractorConfig}" could not be found.`);
        }
        if ((0, e2e_utils_1.isCypressE2eProject)(e2eApp.config)) {
            const configFile = (0, e2e_utils_1.getCypressConfigFile)(e2eApp.config);
            if (configFile && !tree.exists(configFile)) {
                throw new Error(`An e2e project with Cypress was found but "${configFile}" could not be found.`);
            }
            if (!tree.exists('cypress')) {
                throw new Error(`An e2e project with Cypress was found but the "cypress" directory could not be found.`);
            }
            return;
        }
        throw new Error(`An e2e project was found but it's using an unsupported executor "${e2eApp.config.targets.e2e.executor}".`);
    }
    catch (e) {
        console.error(e.message);
        console.error('Your workspace could not be converted into an Nx Workspace because of the above error.');
        throw e;
    }
}
exports.validateWorkspace = validateWorkspace;
function createNxJson(tree, options, setWorkspaceLayoutAsNewProjectRoot = false) {
    var _a;
    const { newProjectRoot = '' } = (0, devkit_1.readJson)(tree, 'angular.json');
    (0, devkit_1.writeJson)(tree, 'nx.json', {
        npmScope: options.npmScope,
        affected: {
            defaultBase: (_a = options.defaultBase) !== null && _a !== void 0 ? _a : (0, default_base_1.deduceDefaultBase)(),
        },
        implicitDependencies: {
            'package.json': {
                dependencies: '*',
                devDependencies: '*',
            },
            '.eslintrc.json': '*',
        },
        tasksRunnerOptions: {
            default: {
                runner: 'nx/tasks-runners/default',
                options: {
                    cacheableOperations: ['build', 'lint', 'test', 'e2e'],
                },
            },
        },
        targetDependencies: {
            build: [
                {
                    target: 'build',
                    projects: 'dependencies',
                },
            ],
        },
        workspaceLayout: setWorkspaceLayoutAsNewProjectRoot
            ? { appsDir: newProjectRoot, libsDir: newProjectRoot }
            : undefined,
    });
}
exports.createNxJson = createNxJson;
function decorateAngularCli(tree) {
    const nrwlWorkspacePath = require.resolve('@nrwl/workspace/package.json');
    const decorateCli = (0, fs_1.readFileSync)((0, path_1.join)((0, path_1.dirname)(nrwlWorkspacePath), 'src/generators/utils/decorate-angular-cli.js__tmpl__'), 'utf-8');
    tree.write('decorate-angular-cli.js', decorateCli);
    (0, devkit_1.updateJson)(tree, 'package.json', (json) => {
        if (json.scripts &&
            json.scripts.postinstall &&
            !json.scripts.postinstall.includes('decorate-angular-cli.js')) {
            // if exists, add execution of this script
            json.scripts.postinstall += ' && node ./decorate-angular-cli.js';
        }
        else {
            if (!json.scripts)
                json.scripts = {};
            // if doesn't exist, set to execute this script
            json.scripts.postinstall = 'node ./decorate-angular-cli.js';
        }
        if (json.scripts.ng) {
            json.scripts.ng = 'nx';
        }
        return json;
    });
}
exports.decorateAngularCli = decorateAngularCli;
function updateWorkspaceConfigDefaults(tree) {
    var _a;
    const workspaceConfig = (0, devkit_1.readWorkspaceConfiguration)(tree);
    delete workspaceConfig.newProjectRoot;
    workspaceConfig.cli = (_a = workspaceConfig.cli) !== null && _a !== void 0 ? _a : {};
    if (!workspaceConfig.cli.defaultCollection) {
        workspaceConfig.cli.defaultCollection = '@nrwl/angular';
    }
    (0, devkit_1.updateWorkspaceConfiguration)(tree, workspaceConfig);
}
exports.updateWorkspaceConfigDefaults = updateWorkspaceConfigDefaults;
function updateRootTsConfig(tree) {
    var _a, _b;
    var _c;
    const tsconfig = (0, devkit_1.readJson)(tree, (0, typescript_1.getRootTsConfigPathInTree)(tree));
    (_a = (_c = tsconfig.compilerOptions).paths) !== null && _a !== void 0 ? _a : (_c.paths = {});
    tsconfig.compilerOptions.baseUrl = '.';
    tsconfig.compilerOptions.rootDir = '.';
    tsconfig.exclude = Array.from(new Set([...((_b = tsconfig.exclude) !== null && _b !== void 0 ? _b : []), 'node_modules', 'tmp']));
    (0, devkit_1.writeJson)(tree, 'tsconfig.base.json', tsconfig);
    if (tree.exists('tsconfig.json')) {
        tree.delete('tsconfig.json');
    }
}
exports.updateRootTsConfig = updateRootTsConfig;
function updatePackageJson(tree) {
    (0, devkit_1.updateJson)(tree, 'package.json', (packageJson) => {
        var _a, _b, _c;
        packageJson.scripts = (_a = packageJson.scripts) !== null && _a !== void 0 ? _a : {};
        Object.keys(packageJson.scripts).forEach((script) => {
            packageJson.scripts[script] = packageJson.scripts[script]
                .replace(/^ng /, 'nx ')
                .replace(/ ng /, ' nx ');
        });
        packageJson.devDependencies = (_b = packageJson.devDependencies) !== null && _b !== void 0 ? _b : {};
        packageJson.dependencies = (_c = packageJson.dependencies) !== null && _c !== void 0 ? _c : {};
        if (!packageJson.devDependencies['@angular/cli']) {
            packageJson.devDependencies['@angular/cli'] = versions_2.angularDevkitVersion;
        }
        if (!packageJson.devDependencies['@nrwl/workspace']) {
            packageJson.devDependencies['@nrwl/workspace'] = versions_2.nxVersion;
        }
        if (!packageJson.devDependencies['nx']) {
            packageJson.devDependencies['nx'] = versions_2.nxVersion;
        }
        if (!packageJson.devDependencies['prettier']) {
            packageJson.devDependencies['prettier'] = versions_1.prettierVersion;
        }
        return packageJson;
    });
}
exports.updatePackageJson = updatePackageJson;
function updateTsLint(tree) {
    if (!tree.exists(`tslint.json`)) {
        return;
    }
    (0, devkit_1.updateJson)(tree, 'tslint.json', (tslintJson) => {
        var _a;
        [
            'no-trailing-whitespace',
            'one-line',
            'quotemark',
            'typedef-whitespace',
            'whitespace',
        ].forEach((key) => {
            tslintJson[key] = undefined;
        });
        tslintJson.rulesDirectory = (_a = tslintJson.rulesDirectory) !== null && _a !== void 0 ? _a : [];
        tslintJson.rulesDirectory.push('node_modules/@nrwl/workspace/src/tslint');
        tslintJson.rules['nx-enforce-module-boundaries'] = [
            true,
            {
                allow: [],
                depConstraints: [{ sourceTag: '*', onlyDependOnLibsWithTags: ['*'] }],
            },
        ];
        return tslintJson;
    });
}
exports.updateTsLint = updateTsLint;
function createWorkspaceFiles(tree) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        updateVsCodeRecommendedExtensions(tree);
        yield updatePrettierConfig(tree);
        (0, devkit_1.generateFiles)(tree, (0, devkit_1.joinPathFragments)(__dirname, '../files/root'), '.', {
            tmpl: '',
            dot: '.',
            rootTsConfigPath: (0, typescript_1.getRootTsConfigPathInTree)(tree),
        });
    });
}
exports.createWorkspaceFiles = createWorkspaceFiles;
function createRootKarmaConfig(tree) {
    (0, devkit_1.generateFiles)(tree, (0, devkit_1.joinPathFragments)(__dirname, '../files/root-karma'), '.', {
        tmpl: '',
    });
}
exports.createRootKarmaConfig = createRootKarmaConfig;
function updateVsCodeRecommendedExtensions(tree) {
    const recommendations = [
        'nrwl.angular-console',
        'angular.ng-template',
        'dbaeumer.vscode-eslint',
        'esbenp.prettier-vscode',
    ];
    if (tree.exists('.vscode/extensions.json')) {
        (0, devkit_1.updateJson)(tree, '.vscode/extensions.json', (json) => {
            json.recommendations = json.recommendations || [];
            recommendations.forEach((extension) => {
                if (!json.recommendations.includes(extension)) {
                    json.recommendations.push(extension);
                }
            });
            return json;
        });
    }
    else {
        (0, devkit_1.writeJson)(tree, '.vscode/extensions.json', {
            recommendations,
        });
    }
}
function updatePrettierConfig(tree) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const existingPrettierConfig = yield (0, prettier_1.resolveUserExistingPrettierConfig)();
        if (!existingPrettierConfig) {
            (0, devkit_1.writeJson)(tree, '.prettierrc', workspace_1.DEFAULT_NRWL_PRETTIER_CONFIG);
        }
        if (!tree.exists('.prettierignore')) {
            (0, devkit_1.generateFiles)(tree, (0, devkit_1.joinPathFragments)(__dirname, '../files/prettier'), '.', { tmpl: '', dot: '.' });
        }
    });
}
//# sourceMappingURL=workspace.js.map