"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.E2eProjectMigrator = void 0;
const tslib_1 = require("tslib");
const devkit_1 = require("@nrwl/devkit");
const typescript_1 = require("@nrwl/workspace/src/utilities/typescript");
const path_1 = require("path");
const e2e_utils_1 = require("./e2e-utils");
const project_migrator_1 = require("./project.migrator");
class E2eProjectMigrator extends project_migrator_1.ProjectMigrator {
    constructor(tree, options, project) {
        super(tree, options, project, 'apps');
        this.appConfig = project.config;
        this.appName = this.project.name;
        this.initializeProject();
    }
    migrate() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!this.project) {
                console.warn('No e2e project was migrated because there was none declared in angular.json.');
                return;
            }
            if ((0, e2e_utils_1.isProtractorE2eProject)(this.projectConfig)) {
                this.migrateProtractorE2eProject();
            }
            else if ((0, e2e_utils_1.isCypressE2eProject)(this.projectConfig)) {
                this.migrateCypressE2eProject();
            }
            const tsConfig = (0, devkit_1.joinPathFragments)(this.projectConfig.root, 'tsconfig.json');
            const rootOffset = (0, devkit_1.offsetFromRoot)(this.project.newRoot);
            (0, devkit_1.updateJson)(this.tree, tsConfig, (json) => {
                json.extends = `${rootOffset}${(0, typescript_1.getRootTsConfigPathInTree)(this.tree)}`;
                json.compilerOptions = Object.assign(Object.assign({}, json.compilerOptions), { outDir: `${rootOffset}dist/out-tsc` });
                return json;
            });
        });
    }
    validate() {
        // TODO: implement validation when multiple apps are supported
        return null;
    }
    initializeProject() {
        var _a;
        if (!((_a = this.projectConfig.targets) === null || _a === void 0 ? void 0 : _a.e2e)) {
            this.project = null;
            return;
        }
        const name = this.project.name.endsWith('-e2e')
            ? this.project.name
            : `${this.project.name}-e2e`;
        const newRoot = (0, devkit_1.joinPathFragments)('apps', name);
        if ((0, e2e_utils_1.isProtractorE2eProject)(this.projectConfig)) {
            this.project = Object.assign(Object.assign({}, this.project), { name, oldRoot: (0, devkit_1.joinPathFragments)(this.project.oldRoot, 'e2e'), newRoot });
        }
        else if ((0, e2e_utils_1.isCypressE2eProject)(this.projectConfig)) {
            this.project = Object.assign(Object.assign({}, this.project), { name, oldRoot: 'cypress', newRoot });
        }
    }
    migrateProtractorE2eProject() {
        this.moveDir(this.project.oldRoot, this.project.newRoot);
        this.projectConfig = {
            root: this.project.newRoot,
            projectType: 'application',
            targets: {
                e2e: Object.assign(Object.assign({}, this.projectConfig.targets.e2e), { options: Object.assign(Object.assign({}, this.projectConfig.targets.e2e.options), { protractorConfig: (0, devkit_1.joinPathFragments)(this.project.newRoot, 'protractor.conf.js') }) }),
                lint: {
                    executor: '@angular-devkit/build-angular:tslint',
                    options: Object.assign(Object.assign({}, this.projectConfig.targets.lint), { tsConfig: (0, devkit_1.joinPathFragments)(this.project.newRoot, 'tsconfig.json') }),
                },
            },
            implicitDependencies: [this.appName],
            tags: [],
        };
        (0, devkit_1.addProjectConfiguration)(this.tree, this.project.name, Object.assign({}, this.projectConfig), true);
        // remove e2e target from the app config
        delete this.appConfig.targets.e2e;
        (0, devkit_1.updateProjectConfiguration)(this.tree, this.appName, Object.assign({}, this.appConfig));
    }
    migrateCypressE2eProject() {
        const oldCypressConfigFilePath = (0, e2e_utils_1.getCypressConfigFile)(this.projectConfig);
        const cypressConfigFilePath = this.updateOrCreateCypressConfigFile(oldCypressConfigFilePath);
        this.moveFile((0, devkit_1.joinPathFragments)(this.project.oldRoot, 'tsconfig.json'), (0, devkit_1.joinPathFragments)(this.project.newRoot, 'tsconfig.json'));
        this.moveDir(this.project.oldRoot, (0, devkit_1.joinPathFragments)(this.project.newRoot, 'src'));
        this.updateCypressProjectConfiguration(cypressConfigFilePath);
    }
    updateOrCreateCypressConfigFile(configFile) {
        let cypressConfigFilePath;
        if (configFile) {
            cypressConfigFilePath = (0, devkit_1.joinPathFragments)(this.project.newRoot, (0, path_1.basename)(configFile));
            this.updateCypressConfigFilePaths(configFile);
            this.moveFile(configFile, cypressConfigFilePath);
        }
        else {
            cypressConfigFilePath = (0, devkit_1.joinPathFragments)(this.project.newRoot, 'cypress.json');
            this.tree.write(cypressConfigFilePath, JSON.stringify({
                fileServerFolder: '.',
                fixturesFolder: './src/fixtures',
                integrationFolder: './src/integration',
                modifyObstructiveCode: false,
                supportFile: './src/support/index.ts',
                pluginsFile: './src/plugins/index.ts',
                video: true,
                videosFolder: `../../dist/cypress/${this.project.newRoot}/videos`,
                screenshotsFolder: `../../dist/cypress/${this.project.newRoot}/screenshots`,
                chromeWebSecurity: false,
            }));
        }
        return cypressConfigFilePath;
    }
    updateCypressProjectConfiguration(cypressConfigPath) {
        this.projectConfig = {
            root: this.project.newRoot,
            sourceRoot: (0, devkit_1.joinPathFragments)(this.project.newRoot, 'src'),
            projectType: 'application',
            targets: {
                e2e: this.updateE2eCypressTarget(this.projectConfig.targets.e2e, cypressConfigPath),
            },
            implicitDependencies: [this.appName],
            tags: [],
        };
        if (this.appConfig.targets['cypress-run']) {
            this.projectConfig.targets['cypress-run'] = this.updateE2eCypressTarget(this.appConfig.targets['cypress-run'], cypressConfigPath);
        }
        if (this.appConfig.targets['cypress-open']) {
            this.projectConfig.targets['cypress-open'] = this.updateE2eCypressTarget(this.appConfig.targets['cypress-open'], cypressConfigPath);
        }
        (0, devkit_1.addProjectConfiguration)(this.tree, this.project.name, Object.assign({}, this.projectConfig), true);
        delete this.appConfig.targets['cypress-run'];
        delete this.appConfig.targets['cypress-open'];
        delete this.appConfig.targets.e2e;
        (0, devkit_1.updateProjectConfiguration)(this.tree, this.appName, Object.assign({}, this.appConfig));
    }
    updateE2eCypressTarget(target, cypressConfig) {
        const updatedTarget = Object.assign(Object.assign({}, target), { executor: '@nrwl/cypress:cypress', options: Object.assign(Object.assign({}, target.options), { cypressConfig }) });
        delete updatedTarget.options.configFile;
        delete updatedTarget.options.tsConfig;
        if (updatedTarget.options.headless && updatedTarget.options.watch) {
            updatedTarget.options.headed = false;
        }
        else if (updatedTarget.options.headless === false &&
            !updatedTarget.options.watch) {
            updatedTarget.options.headed = true;
        }
        delete updatedTarget.options.headless;
        return updatedTarget;
    }
    updateCypressConfigFilePaths(configFilePath) {
        const srcFoldersAndFiles = [
            'integrationFolder',
            'supportFile',
            'pluginsFile',
            'fixturesFolder',
        ];
        const distFolders = ['videosFolder', 'screenshotsFolder'];
        const stringOrArrayGlobs = ['ignoreTestFiles', 'testFiles'];
        const cypressConfig = (0, devkit_1.readJson)(this.tree, configFilePath);
        cypressConfig.fileServerFolder = '.';
        srcFoldersAndFiles.forEach((folderOrFile) => {
            if (cypressConfig[folderOrFile]) {
                cypressConfig[folderOrFile] = `./src/${(0, path_1.relative)(this.project.oldRoot, cypressConfig[folderOrFile])}`;
            }
        });
        distFolders.forEach((folder) => {
            if (cypressConfig[folder]) {
                cypressConfig[folder] = `../../dist/cypress/${this.project.newRoot}/${(0, path_1.relative)(this.project.oldRoot, cypressConfig[folder])}`;
            }
        });
        stringOrArrayGlobs.forEach((stringOrArrayGlob) => {
            if (!cypressConfig[stringOrArrayGlob]) {
                return;
            }
            if (Array.isArray(cypressConfig[stringOrArrayGlob])) {
                cypressConfig[stringOrArrayGlob] = cypressConfig[stringOrArrayGlob].map((glob) => this.replaceCypressGlobConfig(glob));
            }
            else {
                cypressConfig[stringOrArrayGlob] = this.replaceCypressGlobConfig(cypressConfig[stringOrArrayGlob]);
            }
        });
        (0, devkit_1.writeJson)(this.tree, configFilePath, cypressConfig);
    }
    replaceCypressGlobConfig(globPattern) {
        return globPattern.replace(new RegExp(`^(\\.\\/|\\/)?${this.project.oldRoot}\\/`), './src/');
    }
}
exports.E2eProjectMigrator = E2eProjectMigrator;
//# sourceMappingURL=e2e-project.migrator.js.map