"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AppMigrator = void 0;
const tslib_1 = require("tslib");
const devkit_1 = require("@nrwl/devkit");
const generators_1 = require("@nrwl/workspace/generators");
const typescript_1 = require("@nrwl/workspace/src/utilities/typescript");
const e2e_project_migrator_1 = require("./e2e-project.migrator");
const project_migrator_1 = require("./project.migrator");
class AppMigrator extends project_migrator_1.ProjectMigrator {
    constructor(tree, options, project) {
        super(tree, options, project, 'apps');
        this.e2eMigrator = new e2e_project_migrator_1.E2eProjectMigrator(tree, options, project);
    }
    migrate() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            yield this.e2eMigrator.migrate();
            this.moveProjectFiles();
            yield this.updateProjectConfiguration();
            this.updateTsConfigs();
            // TODO: check later if it's still needed
            this.updateProjectTsLint();
        });
    }
    validate() {
        // TODO: implement validation when multiple apps are supported
        return null;
    }
    moveProjectFiles() {
        // it is not required to have a browserslist
        this.moveProjectRootFile((0, devkit_1.joinPathFragments)(this.project.oldRoot, 'browserslist'), false);
        this.moveProjectRootFile((0, devkit_1.joinPathFragments)(this.project.oldRoot, '.browserslistrc'), false);
        this.moveProjectRootFile(this.projectConfig.targets.build.options.tsConfig);
        if (this.projectConfig.targets.test) {
            this.moveProjectRootFile(this.projectConfig.targets.test.options.karmaConfig);
            this.moveProjectRootFile(this.projectConfig.targets.test.options.tsConfig);
        }
        else {
            // there could still be a karma.conf.js file in the root
            // so move to new location
            if (this.tree.exists('karma.conf.js')) {
                console.info('No test configuration, but root Karma config file found');
                this.moveProjectRootFile('karma.conf.js');
            }
        }
        if (this.projectConfig.targets.server) {
            this.moveProjectRootFile(this.projectConfig.targets.server.options.tsConfig);
        }
        this.moveDir(this.project.oldSourceRoot, this.project.newSourceRoot);
    }
    updateProjectConfiguration() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            this.projectConfig.root = this.project.newRoot;
            this.projectConfig.sourceRoot = this.project.newSourceRoot;
            this.convertBuildOptions(this.projectConfig.targets.build.options);
            Object.values(this.projectConfig.targets.build.configurations).forEach((config) => this.convertBuildOptions(config));
            if (this.projectConfig.targets.test) {
                const testOptions = this.projectConfig.targets.test.options;
                testOptions.main =
                    testOptions.main && this.convertAsset(testOptions.main);
                testOptions.polyfills =
                    testOptions.polyfills && this.convertAsset(testOptions.polyfills);
                testOptions.tsConfig = (0, devkit_1.joinPathFragments)(this.project.newRoot, 'tsconfig.spec.json');
                testOptions.karmaConfig = (0, devkit_1.joinPathFragments)(this.project.newRoot, 'karma.conf.js');
                testOptions.assets =
                    testOptions.assets &&
                        testOptions.assets.map((asset) => this.convertAsset(asset));
                testOptions.styles =
                    testOptions.styles &&
                        testOptions.styles.map((style) => this.convertAsset(style));
                testOptions.scripts =
                    testOptions.scripts &&
                        testOptions.scripts.map((script) => this.convertAsset(script));
            }
            if (this.projectConfig.targets.lint) {
                this.projectConfig.targets.lint.options.tsConfig = [
                    (0, devkit_1.joinPathFragments)(this.project.newRoot, 'tsconfig.app.json'),
                    (0, devkit_1.joinPathFragments)(this.project.newRoot, 'tsconfig.spec.json'),
                ];
            }
            if (this.projectConfig.targets.server) {
                const serverOptions = this.projectConfig.targets.server.options;
                this.convertServerOptions(serverOptions);
                Object.values(this.projectConfig.targets.server.configurations).forEach((config) => this.convertServerOptions(config));
            }
            (0, devkit_1.updateProjectConfiguration)(this.tree, this.project.name, Object.assign({}, this.projectConfig));
            yield (0, generators_1.convertToNxProjectGenerator)(this.tree, {
                project: this.project.name,
                skipFormat: true,
            });
        });
    }
    updateTsConfigs() {
        const tsConfigPath = (0, typescript_1.getRootTsConfigPathInTree)(this.tree);
        const projectOffsetFromRoot = (0, devkit_1.offsetFromRoot)(this.projectConfig.root);
        (0, devkit_1.updateJson)(this.tree, this.projectConfig.targets.build.options.tsConfig, (json) => {
            json.extends = `${projectOffsetFromRoot}${tsConfigPath}`;
            json.compilerOptions = json.compilerOptions || {};
            json.compilerOptions.outDir = `${projectOffsetFromRoot}dist/out-tsc`;
            return json;
        });
        if (this.projectConfig.targets.test) {
            (0, devkit_1.updateJson)(this.tree, this.projectConfig.targets.test.options.tsConfig, (json) => {
                var _a;
                json.extends = `${projectOffsetFromRoot}${tsConfigPath}`;
                json.compilerOptions = (_a = json.compilerOptions) !== null && _a !== void 0 ? _a : {};
                json.compilerOptions.outDir = `${projectOffsetFromRoot}dist/out-tsc`;
                return json;
            });
        }
        if (this.projectConfig.targets.server) {
            (0, devkit_1.updateJson)(this.tree, this.projectConfig.targets.server.options.tsConfig, (json) => {
                var _a;
                json.extends = `${projectOffsetFromRoot}${tsConfigPath}`;
                json.compilerOptions = (_a = json.compilerOptions) !== null && _a !== void 0 ? _a : {};
                json.compilerOptions.outDir = `${projectOffsetFromRoot}dist/out-tsc`;
                return json;
            });
        }
    }
    updateProjectTsLint() {
        if (this.tree.exists(`${this.project.newRoot}/tslint.json`)) {
            (0, devkit_1.updateJson)(this.tree, `${this.project.newRoot}/tslint.json`, (json) => {
                json.extends = '../../tslint.json';
                return json;
            });
        }
    }
    convertBuildOptions(buildOptions) {
        buildOptions.outputPath =
            buildOptions.outputPath &&
                (0, devkit_1.joinPathFragments)('dist', this.project.newRoot);
        buildOptions.index =
            buildOptions.index && this.convertAsset(buildOptions.index);
        buildOptions.main =
            buildOptions.main && this.convertAsset(buildOptions.main);
        buildOptions.polyfills =
            buildOptions.polyfills && this.convertAsset(buildOptions.polyfills);
        buildOptions.tsConfig =
            buildOptions.tsConfig &&
                (0, devkit_1.joinPathFragments)(this.project.newRoot, 'tsconfig.app.json');
        buildOptions.assets =
            buildOptions.assets &&
                buildOptions.assets.map((asset) => this.convertAsset(asset));
        buildOptions.styles =
            buildOptions.styles &&
                buildOptions.styles.map((style) => this.convertAsset(style));
        buildOptions.scripts =
            buildOptions.scripts &&
                buildOptions.scripts.map((script) => this.convertAsset(script));
        buildOptions.fileReplacements =
            buildOptions.fileReplacements &&
                buildOptions.fileReplacements.map((replacement) => ({
                    replace: this.convertAsset(replacement.replace),
                    with: this.convertAsset(replacement.with),
                }));
    }
    convertServerOptions(serverOptions) {
        serverOptions.outputPath =
            serverOptions.outputPath &&
                (0, devkit_1.joinPathFragments)('dist', 'apps', `${this.project.name}-server`);
        serverOptions.main =
            serverOptions.main && this.convertAsset(serverOptions.main);
        serverOptions.tsConfig =
            serverOptions.tsConfig &&
                (0, devkit_1.joinPathFragments)(this.project.newRoot, 'tsconfig.server.json');
        serverOptions.fileReplacements =
            serverOptions.fileReplacements &&
                serverOptions.fileReplacements.map((replacement) => ({
                    replace: this.convertAsset(replacement.replace),
                    with: this.convertAsset(replacement.with),
                }));
    }
    convertAsset(asset) {
        if (typeof asset === 'string') {
            return this.convertPath(asset);
        }
        else {
            return Object.assign(Object.assign({}, asset), { input: this.convertPath(asset.input) });
        }
    }
    convertPath(originalPath) {
        return (originalPath === null || originalPath === void 0 ? void 0 : originalPath.startsWith(this.project.oldSourceRoot))
            ? (0, devkit_1.joinPathFragments)(this.project.newSourceRoot, originalPath.replace(this.project.oldSourceRoot, ''))
            : originalPath;
    }
}
exports.AppMigrator = AppMigrator;
//# sourceMappingURL=app.migrator.js.map