"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.updateModuleName = void 0;
const devkit_1 = require("@nrwl/devkit");
const utils_1 = require("@nrwl/workspace/src/generators/move/lib/utils");
const path_1 = require("path");
/**
 * Updates the Angular module name (including the spec file and index.ts)
 *
 * Again, if the user has deviated from the expected folder
 * structure, they are very much on their own.
 *
 * @param schema The options provided to the schematic
 */
function updateModuleName(tree, { projectName, destination }) {
    const newProjectName = (0, utils_1.getNewProjectName)(destination);
    const project = (0, devkit_1.readProjectConfiguration)(tree, newProjectName);
    if (project.projectType === 'application') {
        // Expect the module to be something like 'app.module.ts' regardless of the folder name,
        // Therefore, nothing to do.
        return;
    }
    const moduleName = {
        from: (0, devkit_1.names)(projectName).className,
        to: (0, devkit_1.names)(newProjectName).className,
    };
    const findModuleName = new RegExp(`\\b${moduleName.from}`, 'g');
    const moduleFile = {
        from: `${projectName}.module`,
        to: `${newProjectName}.module`,
    };
    const findFileName = new RegExp(`\\b${moduleFile.from}`, 'g');
    const filesToRename = [
        {
            from: `${project.sourceRoot}/lib/${moduleFile.from}.ts`,
            to: `${project.sourceRoot}/lib/${moduleFile.to}.ts`,
        },
        {
            from: `${project.sourceRoot}/lib/${moduleFile.from}.spec.ts`,
            to: `${project.sourceRoot}/lib/${moduleFile.to}.spec.ts`,
        },
    ].filter((rename) => rename.from !== rename.to);
    if (filesToRename.length === 0) {
        return;
    }
    const replacements = [
        {
            regex: findFileName,
            replaceWith: moduleFile.to,
        },
        {
            regex: findModuleName,
            replaceWith: moduleName.to,
        },
    ];
    // Update the module file and its spec file
    filesToRename.forEach((file) => {
        if (tree.exists(file.from)) {
            updateFileContent(tree, replacements, file.from, file.to);
            tree.delete(file.from);
        }
    });
    // update index file
    const indexFile = (0, path_1.join)(project.sourceRoot, 'index.ts');
    if (tree.exists(indexFile)) {
        updateFileContent(tree, replacements, indexFile);
    }
    const skipFiles = [...filesToRename.map((file) => file.to), indexFile];
    // Update any files which import the module
    for (const [, definition] of (0, devkit_1.getProjects)(tree)) {
        (0, devkit_1.visitNotIgnoredFiles)(tree, definition.root, (file) => {
            // skip files that were already modified
            if (skipFiles.includes(file)) {
                return;
            }
            updateFileContent(tree, replacements, file);
        });
    }
}
exports.updateModuleName = updateModuleName;
function updateFileContent(tree, replacements, fileName, newFileName) {
    var _a;
    let content = (_a = tree.read(fileName)) === null || _a === void 0 ? void 0 : _a.toString('utf-8');
    if (content) {
        let updated = false;
        replacements.forEach((replacement) => {
            if (replacement.regex.test(content)) {
                content = content.replace(replacement.regex, replacement.replaceWith);
                updated = true;
            }
        });
        if (updated) {
            tree.write(newFileName !== null && newFileName !== void 0 ? newFileName : fileName, content);
        }
    }
}
//# sourceMappingURL=update-module-name.js.map