"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.libraryGenerator = void 0;
const tslib_1 = require("tslib");
const devkit_1 = require("@nrwl/devkit");
const ngcli_adapter_1 = require("@nrwl/devkit/ngcli-adapter");
const jest_1 = require("@nrwl/jest");
const linter_1 = require("@nrwl/linter");
const generators_1 = require("@nrwl/workspace/generators");
const init_1 = require("../../generators/init/init");
const versions_1 = require("../../utils/versions");
const add_linting_1 = require("../add-linting/add-linting");
const karma_project_1 = require("../karma-project/karma-project");
const setup_tailwind_1 = require("../setup-tailwind/setup-tailwind");
const add_module_1 = require("./lib/add-module");
const enable_strict_type_checking_1 = require("./lib/enable-strict-type-checking");
const normalize_options_1 = require("./lib/normalize-options");
const update_lib_package_npm_scope_1 = require("./lib/update-lib-package-npm-scope");
const update_project_1 = require("./lib/update-project");
const update_tsconfig_1 = require("./lib/update-tsconfig");
function libraryGenerator(host, schema) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        // Do some validation checks
        if (!schema.routing && schema.lazy) {
            throw new Error(`To use "--lazy" option, "--routing" must also be set.`);
        }
        if (schema.publishable === true && !schema.importPath) {
            throw new Error(`For publishable libs you have to provide a proper "--importPath" which needs to be a valid npm package name (e.g. my-awesome-lib or @myorg/my-lib)`);
        }
        if (schema.addTailwind && !schema.buildable && !schema.publishable) {
            throw new Error(`To use "--addTailwind" option, you have to set either "--buildable" or "--publishable".`);
        }
        const options = (0, normalize_options_1.normalizeOptions)(host, schema);
        yield (0, init_1.default)(host, Object.assign(Object.assign({}, options), { skipFormat: true }));
        const runAngularLibrarySchematic = (0, ngcli_adapter_1.wrapAngularDevkitSchematic)('@schematics/angular', 'library');
        yield runAngularLibrarySchematic(host, {
            name: options.name,
            prefix: options.prefix,
            entryFile: 'index',
            skipPackageJson: !(options.publishable || options.buildable),
            skipTsConfig: true,
        });
        if (options.ngCliSchematicLibRoot !== options.projectRoot) {
            (0, devkit_1.moveFilesToNewDirectory)(host, options.ngCliSchematicLibRoot, options.projectRoot);
        }
        yield (0, update_project_1.updateProject)(host, options);
        (0, update_tsconfig_1.updateTsConfig)(host, options);
        yield addUnitTestRunner(host, options);
        updateNpmScopeIfBuildableOrPublishable(host, options);
        (0, add_module_1.addModule)(host, options);
        setStrictMode(host, options);
        yield addLinting(host, options);
        if (options.addTailwind) {
            yield (0, setup_tailwind_1.default)(host, {
                project: options.name,
                skipFormat: true,
            });
        }
        if (options.buildable || options.publishable) {
            (0, devkit_1.addDependenciesToPackageJson)(host, {}, {
                postcss: versions_1.postcssVersion,
                'postcss-import': '^14.0.2',
                'postcss-preset-env': '^6.7.0',
                'postcss-url': '^10.1.1',
            });
        }
        if (options.standaloneConfig) {
            yield (0, generators_1.convertToNxProjectGenerator)(host, {
                project: options.name,
                all: false,
                skipFormat: true,
            });
        }
        if (!options.skipFormat) {
            yield (0, devkit_1.formatFiles)(host);
        }
        return () => {
            (0, devkit_1.installPackagesTask)(host);
        };
    });
}
exports.libraryGenerator = libraryGenerator;
function addUnitTestRunner(host, options) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        if (options.unitTestRunner === 'jest') {
            yield (0, jest_1.jestProjectGenerator)(host, {
                project: options.name,
                setupFile: 'angular',
                supportTsx: false,
                skipSerializers: false,
                skipFormat: true,
            });
        }
        else if (options.unitTestRunner === 'karma') {
            yield (0, karma_project_1.default)(host, {
                project: options.name,
                skipFormat: true,
            });
        }
    });
}
function updateNpmScopeIfBuildableOrPublishable(host, options) {
    if (options.buildable || options.publishable) {
        (0, update_lib_package_npm_scope_1.updateLibPackageNpmScope)(host, options);
    }
}
function setStrictMode(host, options) {
    if (options.strict) {
        (0, enable_strict_type_checking_1.enableStrictTypeChecking)(host, options);
    }
    else {
        (0, enable_strict_type_checking_1.setLibraryStrictDefault)(host, options.strict);
    }
}
function addLinting(host, options) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        if (options.linter === linter_1.Linter.None) {
            return;
        }
        yield (0, add_linting_1.default)(host, {
            projectName: options.name,
            projectRoot: options.projectRoot,
            prefix: options.prefix,
            setParserOptionsProject: options.setParserOptionsProject,
            skipFormat: true,
        });
    });
}
exports.default = libraryGenerator;
//# sourceMappingURL=library.js.map