"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.conversionSchematic = exports.conversionGenerator = void 0;
const tslib_1 = require("tslib");
const cypress_1 = require("@nrwl/cypress");
const devkit_1 = require("@nrwl/devkit");
const linter_1 = require("@nrwl/linter");
const add_linting_1 = require("../add-linting/add-linting");
function conversionGenerator(host, options) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        /**
         * The ProjectConverter instance encapsulates all the standard operations we need
         * to perform in order to convert a project from TSLint to ESLint, as well as some
         * extensibility points for adjusting the behavior on a per package basis.
         *
         * E.g. @nrwl/angular projects might need to make different changes to the final
         * ESLint config when compared with @nrwl/next projects.
         *
         * See the ProjectConverter implementation for a full breakdown of what it does.
         */
        const projectConverter = new linter_1.ProjectConverter({
            host,
            projectName: options.project,
            ignoreExistingTslintConfig: options.ignoreExistingTslintConfig,
            eslintInitializer: ({ projectName, projectConfig }) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                yield (0, add_linting_1.addLintingGenerator)(host, {
                    projectName,
                    projectRoot: projectConfig.root,
                    prefix: projectConfig.prefix || 'app',
                    /**
                     * We set the parserOptions.project config just in case the converted config uses
                     * rules which require type-checking. Later in the conversion we check if it actually
                     * does and remove the config again if it doesn't, so that it is most efficient.
                     */
                    setParserOptionsProject: true,
                    skipFormat: options.skipFormat,
                });
            }),
        });
        /**
         * If root eslint configuration already exists it will not be recreated
         * but we also don't want to re-run the tslint config conversion
         * as it was likely already done
         */
        const rootEslintConfigExists = host.exists('.eslintrc.json');
        /**
         * Create the standard (which is applicable to the current package) ESLint setup
         * for converting the project.
         */
        const eslintInitInstallTask = yield projectConverter.initESLint();
        /**
         * Convert the root tslint.json and apply the converted rules to the root .eslintrc.json
         */
        const rootConfigInstallTask = yield projectConverter.convertRootTSLintConfig((json) => {
            json.overrides = [
                { files: ['*.ts'], rules: {} },
                { files: ['*.html'], rules: {} },
            ];
            return applyAngularRulesToCorrectOverrides(json);
        }, rootEslintConfigExists);
        /**
         * Convert the project's tslint.json to an equivalent ESLint config.
         */
        const projectConfigInstallTask = yield projectConverter.convertProjectConfig((json) => applyAngularRulesToCorrectOverrides(json));
        /**
         * Clean up the original TSLint configuration for the project.
         */
        projectConverter.removeProjectTSLintFile();
        // Only project shouldn't be added as a default
        const { project } = options, defaults = tslib_1.__rest(options, ["project"]);
        /**
         * Store user preferences for the collection
         */
        projectConverter.setDefaults('@nrwl/angular', defaults);
        /**
         * If the Angular project is an app which has an e2e project, try and convert that as well.
         */
        let cypressInstallTask = () => Promise.resolve(undefined);
        const e2eProjectName = projectConverter.getE2EProjectName();
        if (e2eProjectName) {
            try {
                cypressInstallTask = yield (0, cypress_1.conversionGenerator)(host, {
                    project: e2eProjectName,
                    ignoreExistingTslintConfig: options.ignoreExistingTslintConfig,
                    /**
                     * We can always set this to false, because it will already be handled by the next
                     * step of this parent generator, if applicable
                     */
                    removeTSLintIfNoMoreTSLintTargets: false,
                    skipFormat: options.skipFormat,
                });
            }
            catch (_a) {
                devkit_1.logger.warn('This Angular app has an e2e project, but it was not possible to convert it from TSLint to ESLint. This could be because the e2e project did not have a tslint.json file to begin with.');
            }
        }
        /**
         * Based on user preference and remaining usage, remove TSLint from the workspace entirely.
         */
        let uninstallTSLintTask = () => Promise.resolve(undefined);
        if (options.removeTSLintIfNoMoreTSLintTargets &&
            !projectConverter.isTSLintUsedInWorkspace()) {
            uninstallTSLintTask = projectConverter.removeTSLintFromWorkspace();
        }
        if (!options.skipFormat) {
            yield (0, devkit_1.formatFiles)(host);
        }
        return () => tslib_1.__awaiter(this, void 0, void 0, function* () {
            yield eslintInitInstallTask();
            yield rootConfigInstallTask();
            yield projectConfigInstallTask();
            yield cypressInstallTask();
            yield uninstallTSLintTask();
        });
    });
}
exports.conversionGenerator = conversionGenerator;
exports.conversionSchematic = (0, devkit_1.convertNxGenerator)(conversionGenerator);
/**
 * In the case of Angular lint rules, we need to apply them to correct override depending upon whether
 * or not they require @typescript-eslint/parser or @angular-eslint/template-parser in order to function.
 *
 * By this point, the applicable overrides have already been scaffolded for us by the Nx generators
 * that ran earlier within this generator.
 */
function applyAngularRulesToCorrectOverrides(json) {
    const rules = json.rules;
    if (rules && Object.keys(rules).length) {
        for (const [ruleName, ruleConfig] of Object.entries(rules)) {
            for (const override of json.overrides) {
                if (override.files.includes('*.html') &&
                    ruleName.startsWith('@angular-eslint/template')) {
                    // Prioritize the converted rules over any base implementations from the original Nx generator
                    override.rules[ruleName] = ruleConfig;
                }
                /**
                 * By default, tslint-to-eslint-config will try and apply any rules without known converters
                 * by using eslint-plugin-tslint. We instead explicitly warn the user about this missing converter,
                 * and therefore at this point we strip out any rules which start with @typescript-eslint/tslint/config
                 */
                if (override.files.includes('*.ts') &&
                    !ruleName.startsWith('@angular-eslint/template')) {
                    // Prioritize the converted rules over any base implementations from the original Nx generator
                    override.rules[ruleName] = ruleConfig;
                }
            }
        }
    }
    // It's possible that there are plugins to apply to the TS override
    if (json.plugins) {
        for (const override of json.overrides) {
            if (override.files.includes('*.ts')) {
                override.plugins = override.plugins || [];
                override.plugins = [
                    ...override.plugins,
                    ...json.plugins.filter((plugin) => plugin !== '@angular-eslint/eslint-plugin-template'),
                ];
            }
            if (override.files.includes('*.html') &&
                json.plugins.includes('@angular-eslint/eslint-plugin-template')) {
                override.plugins = ['@angular-eslint/eslint-plugin-template'];
            }
        }
        delete json.plugins;
    }
    /**
     * We now no longer need the flat list of rules at the root of the config
     * because they have all been applied to an appropriate override.
     */
    delete json.rules;
    return json;
}
//# sourceMappingURL=convert-tslint-to-eslint.js.map