"use strict";
/**
 * Adapted from the original ng-packagr source.
 *
 * Changes made:
 * - Changed filePath passed to the StylesheetProcessor.parse when is a .ts file and inlineStyleLanguage is set.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.augmentProgramWithVersioning = exports.cacheCompilerHost = void 0;
const tslib_1 = require("tslib");
const crypto_1 = require("crypto");
const node_1 = require("ng-packagr/lib/graph/node");
const nodes_1 = require("ng-packagr/lib/ng-package/nodes");
const path_1 = require("ng-packagr/lib/utils/path");
const path = require("path");
const ts = require("typescript");
function cacheCompilerHost(graph, entryPoint, compilerOptions, moduleResolutionCache, stylesheetProcessor, inlineStyleLanguage, sourcesFileCache = entryPoint.cache.sourcesFileCache) {
    const compilerHost = ts.createIncrementalCompilerHost(compilerOptions);
    const getNode = (fileName) => {
        const nodeUri = (0, nodes_1.fileUrl)((0, path_1.ensureUnixPath)(fileName));
        let node = graph.get(nodeUri);
        if (!node) {
            node = new node_1.Node(nodeUri);
            graph.put(node);
        }
        return node;
    };
    const addDependee = (fileName) => {
        const node = getNode(fileName);
        entryPoint.dependsOn(node);
    };
    return Object.assign(Object.assign({}, compilerHost), { 
        // ts specific
        fileExists: (fileName) => {
            const cache = sourcesFileCache.getOrCreate(fileName);
            if (cache.exists === undefined) {
                cache.exists = compilerHost.fileExists.call(this, fileName);
            }
            return cache.exists;
        }, getSourceFile: (fileName, languageVersion) => {
            addDependee(fileName);
            const cache = sourcesFileCache.getOrCreate(fileName);
            if (!cache.sourceFile) {
                cache.sourceFile = compilerHost.getSourceFile.call(this, fileName, languageVersion);
            }
            return cache.sourceFile;
        }, writeFile: (fileName, data, writeByteOrderMark, onError, sourceFiles) => {
            if (fileName.endsWith('.d.ts')) {
                sourceFiles.forEach((source) => {
                    const cache = sourcesFileCache.getOrCreate(source.fileName);
                    if (!cache.declarationFileName) {
                        cache.declarationFileName = (0, path_1.ensureUnixPath)(fileName);
                    }
                });
            }
            else {
                fileName = fileName.replace(/\.js(\.map)?$/, '.mjs$1');
                const outputCache = entryPoint.cache.outputCache;
                outputCache.set(fileName, {
                    content: data,
                    version: (0, crypto_1.createHash)('sha256').update(data).digest('hex'),
                });
            }
            compilerHost.writeFile.call(this, fileName, data, writeByteOrderMark, onError, sourceFiles);
        }, readFile: (fileName) => {
            addDependee(fileName);
            const cache = sourcesFileCache.getOrCreate(fileName);
            if (cache.content === undefined) {
                cache.content = compilerHost.readFile.call(this, fileName);
            }
            return cache.content;
        }, resolveModuleNames: (moduleNames, containingFile) => {
            return moduleNames.map((moduleName) => {
                const { resolvedModule } = ts.resolveModuleName(moduleName, (0, path_1.ensureUnixPath)(containingFile), compilerOptions, compilerHost, moduleResolutionCache);
                return resolvedModule;
            });
        }, resourceNameToFileName: (resourceName, containingFilePath) => {
            const resourcePath = path.resolve(path.dirname(containingFilePath), resourceName);
            const containingNode = getNode(containingFilePath);
            const resourceNode = getNode(resourcePath);
            containingNode.dependsOn(resourceNode);
            return resourcePath;
        }, readResource: (fileName) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            addDependee(fileName);
            const cache = sourcesFileCache.getOrCreate(fileName);
            if (cache.content === undefined) {
                if (/(?:html?|svg)$/.test(path.extname(fileName))) {
                    // template
                    cache.content = compilerHost.readFile.call(this, fileName);
                }
                else {
                    // stylesheet
                    cache.content = yield stylesheetProcessor.process({
                        filePath: fileName,
                        content: compilerHost.readFile.call(this, fileName),
                    });
                }
                if (cache.content === undefined) {
                    throw new Error(`Cannot read file ${fileName}.`);
                }
                cache.exists = true;
            }
            return cache.content;
        }), transformResource: (data, context) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (context.resourceFile || context.type !== 'style') {
                return null;
            }
            if (inlineStyleLanguage) {
                const key = (0, crypto_1.createHash)('sha1').update(data).digest('hex');
                const fileName = `${context.containingFile}-${key}.${inlineStyleLanguage}`;
                const cache = sourcesFileCache.getOrCreate(fileName);
                if (cache.content === undefined) {
                    cache.content = yield stylesheetProcessor.process({
                        filePath: context.containingFile,
                        content: data,
                    });
                    const virtualFileNode = getNode(fileName);
                    const containingFileNode = getNode(context.containingFile);
                    virtualFileNode.dependsOn(containingFileNode);
                }
                cache.exists = true;
                return { content: cache.content };
            }
            return null;
        }) });
}
exports.cacheCompilerHost = cacheCompilerHost;
function augmentProgramWithVersioning(program) {
    const baseGetSourceFiles = program.getSourceFiles;
    program.getSourceFiles = function (...parameters) {
        const files = baseGetSourceFiles(...parameters);
        for (const file of files) {
            if (file.version === undefined) {
                file.version = (0, crypto_1.createHash)('sha256').update(file.text).digest('hex');
            }
        }
        return files;
    };
}
exports.augmentProgramWithVersioning = augmentProgramWithVersioning;
//# sourceMappingURL=cache-compiler-host.js.map