"use strict";
/**
 * Adapted from the original ng-packagr source.
 *
 * Changes made:
 * - Added the filePath parameter to the cache key.
 * - Refactored caching to take into account TailwindCSS processing.
 * - Added PostCSS plugins needed to support TailwindCSS.
 * - Added watch mode parameter.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.StylesheetProcessor = exports.InlineStyleLanguage = exports.CssUrl = void 0;
const tslib_1 = require("tslib");
const browserslist = require("browserslist");
const fs_1 = require("fs");
const esbuild_executor_1 = require("ng-packagr/lib/esbuild/esbuild-executor");
const cache_1 = require("ng-packagr/lib/utils/cache");
const log = require("ng-packagr/lib/utils/log");
const path_1 = require("path");
const postcssPresetEnv = require("postcss-preset-env");
const postcssUrl = require("postcss-url");
const tailwindcss_1 = require("../../../utilities/tailwindcss");
const postcss = require('postcss');
var CssUrl;
(function (CssUrl) {
    CssUrl["inline"] = "inline";
    CssUrl["none"] = "none";
})(CssUrl = exports.CssUrl || (exports.CssUrl = {}));
var InlineStyleLanguage;
(function (InlineStyleLanguage) {
    InlineStyleLanguage["sass"] = "sass";
    InlineStyleLanguage["scss"] = "scss";
    InlineStyleLanguage["css"] = "css";
    InlineStyleLanguage["less"] = "less";
})(InlineStyleLanguage = exports.InlineStyleLanguage || (exports.InlineStyleLanguage = {}));
class StylesheetProcessor {
    constructor(basePath, cssUrl, includePaths, cacheDirectory, watch, tailwindConfig) {
        // By default, browserslist defaults are too inclusive
        // https://github.com/browserslist/browserslist/blob/83764ea81ffaa39111c204b02c371afa44a4ff07/index.js#L516-L522
        this.basePath = basePath;
        this.cssUrl = cssUrl;
        this.includePaths = includePaths;
        this.cacheDirectory = cacheDirectory;
        this.watch = watch;
        this.tailwindConfig = tailwindConfig;
        this.esbuild = new esbuild_executor_1.EsbuildExecutor();
        // We change the default query to browsers that Angular support.
        // https://angular.io/guide/browser-support
        browserslist.defaults = [
            'last 1 Chrome version',
            'last 1 Firefox version',
            'last 2 Edge major versions',
            'last 2 Safari major versions',
            'last 2 iOS major versions',
            'Firefox ESR',
        ];
        this.styleIncludePaths = [...this.includePaths];
        let prevDir = null;
        let currentDir = this.basePath;
        while (currentDir !== prevDir) {
            const p = (0, path_1.join)(currentDir, 'node_modules');
            if ((0, fs_1.existsSync)(p)) {
                this.styleIncludePaths.push(p);
            }
            prevDir = currentDir;
            currentDir = (0, path_1.dirname)(prevDir);
        }
        this.browserslistData = browserslist(undefined, { path: this.basePath });
        this.targets = transformSupportedBrowsersToTargets(this.browserslistData);
        this.tailwindSetup = (0, tailwindcss_1.getTailwindSetup)(this.basePath, this.tailwindConfig);
        this.postCssProcessor = this.createPostCssPlugins();
    }
    process({ filePath, content, }) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            let key;
            if (this.cacheDirectory &&
                !content.includes('@import') &&
                !content.includes('@use') &&
                !this.containsTailwindDirectives(content)) {
                // No transitive deps and no Tailwind directives, we can cache more aggressively.
                key = yield (0, cache_1.generateKey)(content, ...this.browserslistData, filePath);
                const result = yield (0, cache_1.readCacheEntry)(this.cacheDirectory, key);
                if (result) {
                    result.warnings.forEach((msg) => log.warn(msg));
                    return result.css;
                }
            }
            // Render pre-processor language (sass, styl, less)
            const renderedCss = yield this.renderCss(filePath, content);
            let containsTailwindDirectives = false;
            if (this.cacheDirectory) {
                containsTailwindDirectives = this.containsTailwindDirectives(renderedCss);
                if (!containsTailwindDirectives) {
                    // No Tailwind directives to process by PostCSS, we can return cached results
                    if (!key) {
                        key = yield (0, cache_1.generateKey)(renderedCss, ...this.browserslistData, filePath);
                    }
                    const cachedResult = yield this.getCachedResult(key);
                    if (cachedResult) {
                        return cachedResult;
                    }
                }
            }
            // Render postcss (autoprefixing and friends)
            const result = yield this.postCssProcessor.process(renderedCss, {
                from: filePath,
                to: filePath.replace((0, path_1.extname)(filePath), '.css'),
            });
            if (this.cacheDirectory && containsTailwindDirectives) {
                // We had Tailwind directives to process by PostCSS, only now
                // is safe to return cached results
                key = yield (0, cache_1.generateKey)(result.css, ...this.browserslistData, filePath);
                const cachedResult = yield this.getCachedResult(key);
                if (cachedResult) {
                    return cachedResult;
                }
            }
            const warnings = result.warnings().map((w) => w.toString());
            const { code, warnings: esBuildWarnings } = yield this.esbuild.transform(result.css, {
                loader: 'css',
                minify: true,
                target: this.targets,
                sourcefile: filePath,
            });
            if (esBuildWarnings.length > 0) {
                warnings.push(...(yield this.esbuild.formatMessages(esBuildWarnings, {
                    kind: 'warning',
                })));
            }
            if (this.cacheDirectory) {
                yield (0, cache_1.saveCacheEntry)(this.cacheDirectory, key, JSON.stringify({
                    css: code,
                    warnings,
                }));
            }
            warnings.forEach((msg) => log.warn(msg));
            return code;
        });
    }
    getCachedResult(key) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const cachedResult = yield (0, cache_1.readCacheEntry)(this.cacheDirectory, key);
            if (cachedResult) {
                cachedResult.warnings.forEach((msg) => log.warn(msg));
                return cachedResult.css;
            }
            return undefined;
        });
    }
    containsTailwindDirectives(content) {
        return (this.tailwindSetup && tailwindcss_1.tailwindDirectives.some((d) => content.includes(d)));
    }
    createPostCssPlugins() {
        const postCssPlugins = [];
        if (this.cssUrl !== CssUrl.none) {
            postCssPlugins.push(postcssUrl({ url: this.cssUrl }));
        }
        if (this.tailwindSetup) {
            postCssPlugins.push(...(0, tailwindcss_1.getTailwindPostCssPlugins)(this.tailwindSetup, this.styleIncludePaths, this.watch));
        }
        postCssPlugins.push(postcssPresetEnv({
            browsers: this.browserslistData,
            autoprefixer: true,
            stage: 3,
        }));
        return postcss(postCssPlugins);
    }
    renderCss(filePath, css) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const ext = (0, path_1.extname)(filePath);
            switch (ext) {
                case '.sass':
                case '.scss': {
                    return (yield Promise.resolve().then(() => require('sass')))
                        .renderSync({
                        file: filePath,
                        data: css,
                        indentedSyntax: '.sass' === ext,
                        importer: customSassImporter,
                        includePaths: this.styleIncludePaths,
                    })
                        .css.toString();
                }
                case '.less': {
                    const { css: content } = yield (yield Promise.resolve().then(() => require('less'))).render(css, {
                        filename: filePath,
                        math: 'always',
                        javascriptEnabled: true,
                        paths: this.styleIncludePaths,
                    });
                    return content;
                }
                case '.styl':
                case '.stylus': {
                    const stylus = (yield Promise.resolve().then(() => require('stylus'))).default;
                    return (stylus(css)
                        // add paths for resolve
                        .set('paths', [
                        this.basePath,
                        '.',
                        ...this.styleIncludePaths,
                        'node_modules',
                    ])
                        // add support for resolving plugins from node_modules
                        .set('filename', filePath)
                        // turn on url resolver in stylus, same as flag --resolve-url
                        .set('resolve url', true)
                        .define('url', stylus.resolver(undefined))
                        .render());
                }
                case '.css':
                default:
                    return css;
            }
        });
    }
}
exports.StylesheetProcessor = StylesheetProcessor;
function transformSupportedBrowsersToTargets(supportedBrowsers) {
    const transformed = [];
    // https://esbuild.github.io/api/#target
    const esBuildSupportedBrowsers = new Set([
        'safari',
        'firefox',
        'edge',
        'chrome',
        'ios',
    ]);
    for (const browser of supportedBrowsers) {
        let [browserName, version] = browser.split(' ');
        // browserslist uses the name `ios_saf` for iOS Safari whereas esbuild uses `ios`
        if (browserName === 'ios_saf') {
            browserName = 'ios';
        }
        // browserslist uses ranges `15.2-15.3` versions but only the lowest is required
        // to perform minimum supported feature checks. esbuild also expects a single version.
        [version] = version.split('-');
        if (browserName === 'ie') {
            transformed.push('edge12');
        }
        else if (esBuildSupportedBrowsers.has(browserName)) {
            if (browserName === 'safari' && version === 'TP') {
                // esbuild only supports numeric versions so `TP` is converted to a high number (999) since
                // a Technology Preview (TP) of Safari is assumed to support all currently known features.
                version = '999';
            }
            transformed.push(browserName + version);
        }
    }
    return transformed.length ? transformed : undefined;
}
function customSassImporter(url, prev) {
    // NB: Sass importer should always be sync as otherwise it will cause
    // sass to go in the async path which is slower.
    if (url[0] !== '~') {
        return undefined;
    }
    return {
        file: url.substring(1),
        prev,
    };
}
//# sourceMappingURL=stylesheet-processor.js.map