"use strict";
/**
 * Adapted from the original ng-packagr.
 *
 * Changes made:
 * - Change the package.json metadata to only use the ESM2020 output as it's the only one generated.
 * - Remove package exports.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.nxWritePackageTransform = void 0;
const tslib_1 = require("tslib");
const devkit_1 = require("@nrwl/devkit");
const node_1 = require("ng-packagr/lib/graph/node");
const transform_1 = require("ng-packagr/lib/graph/transform");
const nodes_1 = require("ng-packagr/lib/ng-package/nodes");
const fs_1 = require("ng-packagr/lib/utils/fs");
const glob_1 = require("ng-packagr/lib/utils/glob");
const path_1 = require("ng-packagr/lib/utils/path");
const path = require("path");
const nxWritePackageTransform = (options) => (0, transform_1.transformFromPromise)((graph) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    var _a;
    const entryPoint = graph.find((0, nodes_1.isEntryPointInProgress)());
    const ngEntryPoint = entryPoint.data.entryPoint;
    const ngPackageNode = graph.find(nodes_1.isPackage);
    const ngPackage = ngPackageNode.data;
    const { destinationFiles } = entryPoint.data;
    const ignorePaths = [
        '.gitkeep',
        '**/.DS_Store',
        '**/Thumbs.db',
        '**/node_modules/**',
        `${ngPackage.dest}/**`,
    ];
    if (!ngEntryPoint.isSecondaryEntryPoint) {
        const assetFiles = [];
        // COPY ASSET FILES TO DESTINATION
        devkit_1.logger.log('Copying assets');
        try {
            for (const asset of ngPackage.assets) {
                let assetFullPath = path.join(ngPackage.src, asset);
                try {
                    const stats = yield (0, fs_1.stat)(assetFullPath);
                    if (stats.isFile()) {
                        assetFiles.push(assetFullPath);
                        continue;
                    }
                    if (stats.isDirectory()) {
                        assetFullPath = path.join(assetFullPath, '**/*');
                    }
                }
                catch (_b) { }
                const files = yield (0, glob_1.globFiles)(assetFullPath, {
                    ignore: ignorePaths,
                    cache: ngPackageNode.cache.globCache,
                    dot: true,
                    nodir: true,
                });
                if (files.length) {
                    assetFiles.push(...files);
                }
            }
            for (const file of assetFiles) {
                const relativePath = path.relative(ngPackage.src, file);
                const destination = path.resolve(ngPackage.dest, relativePath);
                const nodeUri = (0, nodes_1.fileUrl)((0, path_1.ensureUnixPath)(file));
                let node = graph.get(nodeUri);
                if (!node) {
                    node = new node_1.Node(nodeUri);
                    graph.put(node);
                }
                entryPoint.dependsOn(node);
                yield (0, fs_1.copyFile)(file, destination);
            }
        }
        catch (error) {
            throw error;
        }
    }
    // 6. WRITE PACKAGE.JSON
    try {
        devkit_1.logger.info('Writing package metadata');
        const relativeUnixFromDestPath = (filePath) => (0, path_1.ensureUnixPath)(path.relative(ngEntryPoint.destinationPath, filePath));
        yield writePackageJson(ngEntryPoint, ngPackage, {
            module: relativeUnixFromDestPath(destinationFiles.esm2020),
            es2020: relativeUnixFromDestPath(destinationFiles.esm2020),
            esm2020: relativeUnixFromDestPath(destinationFiles.esm2020),
            typings: relativeUnixFromDestPath(destinationFiles.declarations),
            // webpack v4+ specific flag to enable advanced optimizations and code splitting
            sideEffects: (_a = ngEntryPoint.packageJson.sideEffects) !== null && _a !== void 0 ? _a : false,
        }, !!options.watch);
    }
    catch (error) {
        throw error;
    }
    devkit_1.logger.info(`Built ${ngEntryPoint.moduleId}`);
    return graph;
}));
exports.nxWritePackageTransform = nxWritePackageTransform;
/**
 * Creates and writes a `package.json` file of the entry point used by the `node_module`
 * resolution strategies.
 *
 * #### Example
 *
 * A consumer of the entry point depends on it by `import {..} from '@my/module/id';`.
 * The module id `@my/module/id` will be resolved to the `package.json` file that is written by
 * this build step.
 * The properties `main`, `module`, `typings` (and so on) in the `package.json` point to the
 * flattened JavaScript bundles, type definitions, (...).
 *
 * @param entryPoint An entry point of an Angular package / library
 * @param additionalProperties Additional properties, e.g. binary artefacts (bundle files), to merge into `package.json`
 */
function writePackageJson(entryPoint, pkg, additionalProperties, isWatchMode) {
    var _a, _b, _c;
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        // set additional properties
        const packageJson = Object.assign(Object.assign({}, entryPoint.packageJson), additionalProperties);
        // read tslib version from `@angular/compiler` so that our tslib
        // version at least matches that of angular if we use require('tslib').version
        // it will get what installed and not the minimum version nor if it is a `~` or `^`
        // this is only required for primary
        if (!entryPoint.isSecondaryEntryPoint) {
            if (isWatchMode) {
                // Needed because of Webpack's 5 `cachemanagedpaths`
                // https://github.com/angular/angular-cli/issues/20962
                packageJson.version = `0.0.0-watch+${Date.now()}`;
            }
            if (!((_a = packageJson.peerDependencies) === null || _a === void 0 ? void 0 : _a.tslib) &&
                !((_b = packageJson.dependencies) === null || _b === void 0 ? void 0 : _b.tslib)) {
                const { peerDependencies: angularPeerDependencies = {}, dependencies: angularDependencies = {}, } = require('@angular/compiler/package.json');
                const tsLibVersion = angularPeerDependencies.tslib || angularDependencies.tslib;
                if (tsLibVersion) {
                    packageJson.dependencies = Object.assign(Object.assign({}, packageJson.dependencies), { tslib: tsLibVersion });
                }
            }
            else if ((_c = packageJson.peerDependencies) === null || _c === void 0 ? void 0 : _c.tslib) {
                devkit_1.logger.warn(`'tslib' is no longer recommended to be used as a 'peerDependencies'. Moving it to 'dependencies'.`);
                packageJson.dependencies = Object.assign(Object.assign({}, (packageJson.dependencies || {})), { tslib: packageJson.peerDependencies.tslib });
                delete packageJson.peerDependencies.tslib;
            }
        }
        // Verify non-peerDependencies as they can easily lead to duplicate installs or version conflicts
        // in the node_modules folder of an application
        const allowedList = pkg.allowedNonPeerDependencies.map((value) => new RegExp(value));
        try {
            checkNonPeerDependencies(packageJson, 'dependencies', allowedList);
        }
        catch (e) {
            yield (0, fs_1.rmdir)(entryPoint.destinationPath, { recursive: true });
            throw e;
        }
        // Removes scripts from package.json after build
        if (packageJson.scripts) {
            if (pkg.keepLifecycleScripts !== true) {
                devkit_1.logger.info(`Removing scripts section in package.json as it's considered a potential security vulnerability.`);
                delete packageJson.scripts;
            }
            else {
                devkit_1.logger.warn(`You enabled keepLifecycleScripts explicitly. The scripts section in package.json will be published to npm.`);
            }
        }
        // keep the dist package.json clean
        // this will not throw if ngPackage field does not exist
        delete packageJson.ngPackage;
        const packageJsonPropertiesToDelete = [
            'stylelint',
            'prettier',
            'browserslist',
            'devDependencies',
            'jest',
            'workspaces',
            'husky',
        ];
        for (const prop of packageJsonPropertiesToDelete) {
            if (prop in packageJson) {
                delete packageJson[prop];
                devkit_1.logger.info(`Removing ${prop} section in package.json.`);
            }
        }
        packageJson.name = entryPoint.moduleId;
        yield (0, fs_1.writeFile)(path.join(entryPoint.destinationPath, 'package.json'), JSON.stringify(packageJson, undefined, 2));
    });
}
function checkNonPeerDependencies(packageJson, property, allowed) {
    if (!packageJson[property]) {
        return;
    }
    for (const dep of Object.keys(packageJson[property])) {
        if (!allowed.some((regex) => regex.test(dep))) {
            devkit_1.logger.warn(`Distributing npm packages with '${property}' is not recommended. Please consider adding ${dep} to 'peerDependencies' or remove it from '${property}'.`);
            throw new Error(`Dependency ${dep} must be explicitly allowed using the "allowedNonPeerDependencies" option.`);
        }
    }
}
//# sourceMappingURL=write-package.transform.js.map