"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.setupEnvironment = exports.addMatchers = exports.time = exports.cold = exports.hot = exports.resetTestScheduler = exports.initTestScheduler = exports.getTestScheduler = void 0;
var rxjs_1 = require("rxjs");
var testing_1 = require("rxjs/testing");
var lodash_1 = require("lodash");
var scheduler_1 = require("./src/scheduler");
var test_observables_1 = require("./src/test-observables");
var marble_unparser_1 = require("./src/marble-unparser");
var map_symbols_to_notifications_1 = require("./src/map-symbols-to-notifications");
var scheduler_2 = require("./src/scheduler");
Object.defineProperty(exports, "getTestScheduler", { enumerable: true, get: function () { return scheduler_2.getTestScheduler; } });
Object.defineProperty(exports, "initTestScheduler", { enumerable: true, get: function () { return scheduler_2.initTestScheduler; } });
Object.defineProperty(exports, "resetTestScheduler", { enumerable: true, get: function () { return scheduler_2.resetTestScheduler; } });
function hot(marbles, values, error) {
    return new test_observables_1.TestHotObservable(marbles.trim(), values, error);
}
exports.hot = hot;
function cold(marbles, values, error) {
    return new test_observables_1.TestColdObservable(marbles.trim(), values, error);
}
exports.cold = cold;
function time(marbles) {
    return scheduler_1.getTestScheduler().createTime(marbles.trim());
}
exports.time = time;
/*
 * Based on source code found in rxjs library
 * https://github.com/ReactiveX/rxjs/blob/master/src/testing/TestScheduler.ts
 *
 */
function materializeInnerObservable(observable, outerFrame) {
    var messages = [];
    var scheduler = scheduler_1.getTestScheduler();
    observable.subscribe(function (value) {
        messages.push({
            frame: scheduler.frame - outerFrame,
            notification: rxjs_1.Notification.createNext(value),
        });
    }, function (err) {
        messages.push({
            frame: scheduler.frame - outerFrame,
            notification: rxjs_1.Notification.createError(err),
        });
    }, function () {
        messages.push({
            frame: scheduler.frame - outerFrame,
            notification: rxjs_1.Notification.createComplete(),
        });
    });
    return messages;
}
var toHaveSubscriptionsComparer = function (actual, marbles) {
    var marblesArray = typeof marbles === 'string' ? [marbles] : marbles;
    var results = marblesArray.map(function (marbles) {
        return testing_1.TestScheduler.parseMarblesAsSubscriptions(marbles);
    });
    expect(results).toEqual(actual.getSubscriptions());
    return { pass: true, message: function () { return ''; } };
};
var toBeObservableComparer = function (actual, fixture) {
    var results = [];
    var subscription;
    var scheduler = scheduler_1.getTestScheduler();
    scheduler.schedule(function () {
        subscription = actual.subscribe(function (x) {
            var value = x;
            // Support Observable-of-Observables
            if (x instanceof rxjs_1.Observable) {
                value = materializeInnerObservable(value, scheduler.frame);
            }
            results.push({
                frame: scheduler.frame,
                notification: rxjs_1.Notification.createNext(value),
            });
        }, function (err) {
            results.push({
                frame: scheduler.frame,
                notification: rxjs_1.Notification.createError(err),
            });
        }, function () {
            results.push({
                frame: scheduler.frame,
                notification: rxjs_1.Notification.createComplete(),
            });
        });
    });
    scheduler.flush();
    var expected = testing_1.TestScheduler.parseMarbles(fixture.marbles, fixture.values, fixture.error, true, true);
    try {
        expect(results).toEqual(expected);
        return { pass: true, message: function () { return ''; } };
    }
    catch (e) {
        var mapNotificationToSymbol = buildNotificationToSymbolMapper(fixture.marbles, expected, lodash_1.isEqual);
        var receivedMarble = marble_unparser_1.unparseMarble(results, mapNotificationToSymbol);
        var message_1 = formatMessage(fixture.marbles, expected, receivedMarble, results);
        return { pass: false, message: function () { return message_1; } };
    }
};
function addMatchers() {
    if (!expect.extend) {
        jasmine.addMatchers({
            toHaveSubscriptions: function () { return ({
                compare: toHaveSubscriptionsComparer,
            }); },
            toBeObservable: function (utils, equalityTester) { return ({
                compare: toBeObservableComparer,
            }); },
        });
    }
    else {
        expect.extend({
            toHaveSubscriptions: toHaveSubscriptionsComparer,
            toBeObservable: toBeObservableComparer,
        });
    }
}
exports.addMatchers = addMatchers;
function buildNotificationToSymbolMapper(expectedMarbles, expectedMessages, equalityFn) {
    var symbolsToNotificationsMap = map_symbols_to_notifications_1.mapSymbolsToNotifications(expectedMarbles, expectedMessages);
    return function (notification) {
        var mapped = Object.keys(symbolsToNotificationsMap).find(function (key) {
            return equalityFn(symbolsToNotificationsMap[key], notification);
        });
        return mapped || '?';
    };
}
function formatMessage(expectedMarbles, expectedMessages, receivedMarbles, receivedMessages) {
    return "\n    Expected: " + expectedMarbles + ",\n    Received: " + receivedMarbles + ",\n    \n    Expected:\n    " + JSON.stringify(expectedMessages) + "\n    \n    Received:\n    " + JSON.stringify(receivedMessages) + ",\n  ";
}
function setupEnvironment() {
    beforeAll(function () { return addMatchers(); });
    beforeEach(function () { return scheduler_1.initTestScheduler(); });
    afterEach(function () {
        scheduler_1.getTestScheduler().flush();
        scheduler_1.resetTestScheduler();
    });
}
exports.setupEnvironment = setupEnvironment;
setupEnvironment();
//# sourceMappingURL=index.js.map