import { Notification, Observable } from 'rxjs';
import { TestScheduler } from 'rxjs/testing';
import { isEqual } from 'lodash';
import { getTestScheduler, initTestScheduler, resetTestScheduler, } from './src/scheduler';
import { TestColdObservable, TestHotObservable, } from './src/test-observables';
import { unparseMarble } from './src/marble-unparser';
import { mapSymbolsToNotifications } from './src/map-symbols-to-notifications';
export { getTestScheduler, initTestScheduler, resetTestScheduler, } from './src/scheduler';
export function hot(marbles, values, error) {
    return new TestHotObservable(marbles.trim(), values, error);
}
export function cold(marbles, values, error) {
    return new TestColdObservable(marbles.trim(), values, error);
}
export function time(marbles) {
    return getTestScheduler().createTime(marbles.trim());
}
/*
 * Based on source code found in rxjs library
 * https://github.com/ReactiveX/rxjs/blob/master/src/testing/TestScheduler.ts
 *
 */
function materializeInnerObservable(observable, outerFrame) {
    const messages = [];
    const scheduler = getTestScheduler();
    observable.subscribe((value) => {
        messages.push({
            frame: scheduler.frame - outerFrame,
            notification: Notification.createNext(value),
        });
    }, (err) => {
        messages.push({
            frame: scheduler.frame - outerFrame,
            notification: Notification.createError(err),
        });
    }, () => {
        messages.push({
            frame: scheduler.frame - outerFrame,
            notification: Notification.createComplete(),
        });
    });
    return messages;
}
const toHaveSubscriptionsComparer = function (actual, marbles) {
    const marblesArray = typeof marbles === 'string' ? [marbles] : marbles;
    const results = marblesArray.map((marbles) => TestScheduler.parseMarblesAsSubscriptions(marbles));
    expect(results).toEqual(actual.getSubscriptions());
    return { pass: true, message: () => '' };
};
const toBeObservableComparer = function (actual, fixture) {
    const results = [];
    let subscription;
    const scheduler = getTestScheduler();
    scheduler.schedule(() => {
        subscription = actual.subscribe((x) => {
            let value = x;
            // Support Observable-of-Observables
            if (x instanceof Observable) {
                value = materializeInnerObservable(value, scheduler.frame);
            }
            results.push({
                frame: scheduler.frame,
                notification: Notification.createNext(value),
            });
        }, (err) => {
            results.push({
                frame: scheduler.frame,
                notification: Notification.createError(err),
            });
        }, () => {
            results.push({
                frame: scheduler.frame,
                notification: Notification.createComplete(),
            });
        });
    });
    scheduler.flush();
    const expected = TestScheduler.parseMarbles(fixture.marbles, fixture.values, fixture.error, true, true);
    try {
        expect(results).toEqual(expected);
        return { pass: true, message: () => '' };
    }
    catch (e) {
        const mapNotificationToSymbol = buildNotificationToSymbolMapper(fixture.marbles, expected, isEqual);
        const receivedMarble = unparseMarble(results, mapNotificationToSymbol);
        const message = formatMessage(fixture.marbles, expected, receivedMarble, results);
        return { pass: false, message: () => message };
    }
};
export function addMatchers() {
    if (!expect.extend) {
        jasmine.addMatchers({
            toHaveSubscriptions: () => ({
                compare: toHaveSubscriptionsComparer,
            }),
            toBeObservable: (utils, equalityTester) => ({
                compare: toBeObservableComparer,
            }),
        });
    }
    else {
        expect.extend({
            toHaveSubscriptions: toHaveSubscriptionsComparer,
            toBeObservable: toBeObservableComparer,
        });
    }
}
function buildNotificationToSymbolMapper(expectedMarbles, expectedMessages, equalityFn) {
    const symbolsToNotificationsMap = mapSymbolsToNotifications(expectedMarbles, expectedMessages);
    return (notification) => {
        const mapped = Object.keys(symbolsToNotificationsMap).find((key) => {
            return equalityFn(symbolsToNotificationsMap[key], notification);
        });
        return mapped || '?';
    };
}
function formatMessage(expectedMarbles, expectedMessages, receivedMarbles, receivedMessages) {
    return `
    Expected: ${expectedMarbles},
    Received: ${receivedMarbles},
    
    Expected:
    ${JSON.stringify(expectedMessages)}
    
    Received:
    ${JSON.stringify(receivedMessages)},
  `;
}
export function setupEnvironment() {
    beforeAll(() => addMatchers());
    beforeEach(() => initTestScheduler());
    afterEach(() => {
        getTestScheduler().flush();
        resetTestScheduler();
    });
}
setupEnvironment();
//# sourceMappingURL=index.js.map