import { Injectable } from '@angular/core';
import { ofType } from '@ngrx/effects';
import { ROUTER_NAVIGATION } from '@ngrx/router-store';
import { isObservable, of } from 'rxjs';
import { catchError, concatMap, filter, groupBy, map, mergeMap, switchMap, withLatestFrom, } from 'rxjs/operators';
import * as i0 from "@angular/core";
import * as i1 from "@ngrx/store";
import * as i2 from "@ngrx/effects";
export function pessimisticUpdate(opts) {
    return (source) => {
        return source.pipe(mapActionAndState(), concatMap(runWithErrorHandling(opts.run, opts.onError)));
    };
}
export function optimisticUpdate(opts) {
    return (source) => {
        return source.pipe(mapActionAndState(), concatMap(runWithErrorHandling(opts.run, opts.undoAction)));
    };
}
export function fetch(opts) {
    return (source) => {
        if (opts.id) {
            const groupedFetches = source.pipe(mapActionAndState(), groupBy(([action, ...store]) => {
                return opts.id(action, ...store);
            }));
            return groupedFetches.pipe(mergeMap((pairs) => pairs.pipe(switchMap(runWithErrorHandling(opts.run, opts.onError)))));
        }
        return source.pipe(mapActionAndState(), concatMap(runWithErrorHandling(opts.run, opts.onError)));
    };
}
export function navigation(component, opts) {
    return (source) => {
        const nav = source.pipe(mapActionAndState(), filter(([action]) => isStateSnapshot(action)), map(([action, ...slices]) => {
            if (!isStateSnapshot(action)) {
                // Because of the above filter we'll never get here,
                // but this properly type narrows `action`
                return;
            }
            return [
                findSnapshot(component, action.payload.routerState.root),
                ...slices,
            ];
        }), filter(([snapshot]) => !!snapshot));
        return nav.pipe(switchMap(runWithErrorHandling(opts.run, opts.onError)));
    };
}
function isStateSnapshot(action) {
    return action.type === ROUTER_NAVIGATION;
}
function runWithErrorHandling(run, onError) {
    return ([action, ...slices]) => {
        try {
            const r = wrapIntoObservable(run(action, ...slices));
            return r.pipe(catchError((e) => wrapIntoObservable(onError(action, e))));
        }
        catch (e) {
            return wrapIntoObservable(onError(action, e));
        }
    };
}
/**
 * @whatItDoes maps Observable<Action | [Action, State]> to
 * Observable<[Action, State]>
 */
function mapActionAndState() {
    return (source) => {
        return source.pipe(map((value) => normalizeActionAndState(value)));
    };
}
/**
 * @whatItDoes Normalizes either a bare action or an array of action and slices
 * into an array of action and slices (or undefined)
 */
function normalizeActionAndState(args) {
    let action, slices;
    if (args instanceof Array) {
        [action, ...slices] = args;
    }
    else {
        slices = [];
        action = args;
    }
    return [action, ...slices];
}
/**
 * @whatItDoes Provides convenience methods for implementing common operations of persisting data.
 */
export class DataPersistence {
    constructor(store, actions) {
        this.store = store;
        this.actions = actions;
    }
    /**
     *
     * @whatItDoes Handles pessimistic updates (updating the server first).
     *
     * Update the server implemented naively suffers from race conditions and poor error handling.
     *
     * `pessimisticUpdate` addresses these problems--it runs all fetches in order, which removes race conditions
     * and forces the developer to handle errors.
     *
     * ## Example:
     *
     * ```typescript
     * @Injectable()
     * class TodoEffects {
     *   @Effect() updateTodo = this.s.pessimisticUpdate<UpdateTodo>('UPDATE_TODO', {
     *     // provides an action and the current state of the store
     *     run(a, state) {
     *       // update the backend first, and then dispatch an action that will
     *       // update the client side
     *       return this.backend(state.user, a.payload).map(updated => ({
     *         type: 'TODO_UPDATED',
     *         payload: updated
     *       }));
     *     },
     *
     *     onError(a, e: any) {
     *       // we don't need to undo the changes on the client side.
     *       // we can dispatch an error, or simply log the error here and return `null`
     *       return null;
     *     }
     *   });
     *
     *   constructor(private s: DataPersistence<TodosState>, private backend: Backend) {}
     * }
     * ```
     *
     * Note that if you don't return a new action from the run callback, you must set the dispatch property
     * of the effect to false, like this:
     *
     * ```
     * class TodoEffects {
     *   @Effect({dispatch: false})
     *   updateTodo; //...
     * }
     * ```
     */
    pessimisticUpdate(actionType, opts) {
        return this.actions.pipe(ofType(actionType), withLatestFrom(this.store), pessimisticUpdate(opts));
    }
    /**
     *
     * @whatItDoes Handles optimistic updates (updating the client first).
     *
     * `optimisticUpdate` addresses these problems--it runs all fetches in order, which removes race conditions
     * and forces the developer to handle errors.
     *
     * `optimisticUpdate` is different from `pessimisticUpdate`. In case of a failure, when using `optimisticUpdate`,
     * the developer already updated the state locally, so the developer must provide an undo action.
     *
     * The error handling must be done in the callback, or by means of the undo action.
     *
     * ## Example:
     *
     * ```typescript
     * @Injectable()
     * class TodoEffects {
     *   @Effect() updateTodo = this.s.optimisticUpdate<UpdateTodo>('UPDATE_TODO', {
     *     // provides an action and the current state of the store
     *     run: (a, state) => {
     *       return this.backend(state.user, a.payload);
     *     },
     *
     *     undoAction: (a, e: any) => {
     *       // dispatch an undo action to undo the changes in the client state
     *       return ({
     *         type: 'UNDO_UPDATE_TODO',
     *         payload: a
     *       });
     *     }
     *   });
     *
     *   constructor(private s: DataPersistence<TodosState>, private backend: Backend) {}
     * }
     * ```
     *
     * Note that if you don't return a new action from the run callback, you must set the dispatch property
     * of the effect to false, like this:
     *
     * ```
     * class TodoEffects {
     *   @Effect({dispatch: false})
     *   updateTodo; //...
     * }
     * ```
     */
    optimisticUpdate(actionType, opts) {
        return this.actions.pipe(ofType(actionType), withLatestFrom(this.store), optimisticUpdate(opts));
    }
    /**
     *
     * @whatItDoes Handles data fetching.
     *
     * Data fetching implemented naively suffers from race conditions and poor error handling.
     *
     * `fetch` addresses these problems--it runs all fetches in order, which removes race conditions
     * and forces the developer to handle errors.
     *
     * ## Example:
     *
     * ```typescript
     * @Injectable()
     * class TodoEffects {
     *   @Effect() loadTodos = this.s.fetch<GetTodos>('GET_TODOS', {
     *     // provides an action and the current state of the store
     *     run: (a, state) => {
     *       return this.backend(state.user, a.payload).map(r => ({
     *         type: 'TODOS',
     *         payload: r
     *       });
     *     },
     *
     *     onError: (a, e: any) => {
     *       // dispatch an undo action to undo the changes in the client state
     *     }
     *   });
     *
     *   constructor(private s: DataPersistence<TodosState>, private backend: Backend) {}
     * }
     * ```
     *
     * This is correct, but because it set the concurrency to 1, it may not be performant.
     *
     * To fix that, you can provide the `id` function, like this:
     *
     * ```typescript
     * @Injectable()
     * class TodoEffects {
     *   @Effect() loadTodo = this.s.fetch<GetTodo>('GET_TODO', {
     *     id: (a, state) => {
     *       return a.payload.id;
     *     }
     *
     *     // provides an action and the current state of the store
     *     run: (a, state) => {
     *       return this.backend(state.user, a.payload).map(r => ({
     *         type: 'TODO',
     *         payload: r
     *       });
     *     },
     *
     *     onError: (a, e: any) => {
     *       // dispatch an undo action to undo the changes in the client state
     *       return null;
     *     }
     *   });
     *
     *   constructor(private s: DataPersistence<TodosState>, private backend: Backend) {}
     * }
     * ```
     *
     * With this setup, the requests for Todo 1 will run concurrently with the requests for Todo 2.
     *
     * In addition, if DataPersistence notices that there are multiple requests for Todo 1 scheduled,
     * it will only run the last one.
     */
    fetch(actionType, opts) {
        return this.actions.pipe(ofType(actionType), withLatestFrom(this.store), fetch(opts));
    }
    /**
     * @whatItDoes Handles data fetching as part of router navigation.
     *
     * Data fetching implemented naively suffers from race conditions and poor error handling.
     *
     * `navigation` addresses these problems.
     *
     * It checks if an activated router state contains the passed in component type, and, if it does, runs the `run`
     * callback. It provides the activated snapshot associated with the component and the current state. And it only runs
     * the last request.
     *
     * ## Example:
     *
     * ```typescript
     * @Injectable()
     * class TodoEffects {
     *   @Effect() loadTodo = this.s.navigation(TodoComponent, {
     *     run: (a, state) => {
     *       return this.backend.fetchTodo(a.params['id']).map(todo => ({
     *         type: 'TODO_LOADED',
     *         payload: todo
     *       }));
     *     },
     *     onError: (a, e: any) => {
     *       // we can log and error here and return null
     *       // we can also navigate back
     *       return null;
     *     }
     *   });
     *   constructor(private s: DataPersistence<TodosState>, private backend: Backend) {}
     * }
     * ```
     */
    navigation(component, opts) {
        return this.actions.pipe(withLatestFrom(this.store), navigation(component, opts));
    }
}
DataPersistence.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "13.3.0", ngImport: i0, type: DataPersistence, deps: [{ token: i1.Store }, { token: i2.Actions }], target: i0.ɵɵFactoryTarget.Injectable });
DataPersistence.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "13.3.0", ngImport: i0, type: DataPersistence });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "13.3.0", ngImport: i0, type: DataPersistence, decorators: [{
            type: Injectable
        }], ctorParameters: function () { return [{ type: i1.Store }, { type: i2.Actions }]; } });
function findSnapshot(component, s) {
    if (s.routeConfig && s.routeConfig.component === component) {
        return s;
    }
    for (const c of s.children) {
        const ss = findSnapshot(component, c);
        if (ss) {
            return ss;
        }
    }
    return null;
}
function wrapIntoObservable(obj) {
    if (isObservable(obj)) {
        return obj;
    }
    else if (!obj) {
        return of();
    }
    else {
        return of(obj);
    }
}
//# sourceMappingURL=data:application/json;base64,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