import { isDevMode } from '@angular/core';
import { isNgrxMockEnvironment } from './flags';
export function isEqualCheck(a, b) {
    return a === b;
}
function isArgumentsChanged(args, lastArguments, comparator) {
    for (let i = 0; i < args.length; i++) {
        if (!comparator(args[i], lastArguments[i])) {
            return true;
        }
    }
    return false;
}
export function resultMemoize(projectionFn, isResultEqual) {
    return defaultMemoize(projectionFn, isEqualCheck, isResultEqual);
}
export function defaultMemoize(projectionFn, isArgumentsEqual = isEqualCheck, isResultEqual = isEqualCheck) {
    let lastArguments = null;
    // eslint-disable-next-line @typescript-eslint/no-explicit-any, , , , ,
    let lastResult = null;
    let overrideResult;
    function reset() {
        lastArguments = null;
        lastResult = null;
    }
    function setResult(result = undefined) {
        overrideResult = { result };
    }
    function clearResult() {
        overrideResult = undefined;
    }
    /* eslint-disable prefer-rest-params, prefer-spread */
    // disabled because of the use of `arguments`
    function memoized() {
        if (overrideResult !== undefined) {
            return overrideResult.result;
        }
        if (!lastArguments) {
            lastResult = projectionFn.apply(null, arguments);
            lastArguments = arguments;
            return lastResult;
        }
        if (!isArgumentsChanged(arguments, lastArguments, isArgumentsEqual)) {
            return lastResult;
        }
        const newResult = projectionFn.apply(null, arguments);
        lastArguments = arguments;
        if (isResultEqual(lastResult, newResult)) {
            return lastResult;
        }
        lastResult = newResult;
        return newResult;
    }
    return { memoized, reset, setResult, clearResult };
}
export function createSelector(...input) {
    return createSelectorFactory(defaultMemoize)(...input);
}
export function defaultStateFn(state, selectors, props, memoizedProjector) {
    if (props === undefined) {
        const args = selectors.map((fn) => fn(state));
        return memoizedProjector.memoized.apply(null, args);
    }
    const args = selectors.map((fn) => fn(state, props));
    return memoizedProjector.memoized.apply(null, [...args, props]);
}
/**
 *
 * @param memoize The function used to memoize selectors
 * @param options Config Object that may include a `stateFn` function defining how to return the selector's value, given the entire `Store`'s state, parent `Selector`s, `Props`, and a `MemoizedProjection`
 *
 * @usageNotes
 *
 * **Creating a Selector Factory Where Array Order Does Not Matter**
 *
 * ```ts
 * function removeMatch(arr: string[], target: string): string[] {
 *   const matchIndex = arr.indexOf(target);
 *   return [...arr.slice(0, matchIndex), ...arr.slice(matchIndex + 1)];
 * }
 *
 * function orderDoesNotMatterComparer(a: any, b: any): boolean {
 *   if (!Array.isArray(a) || !Array.isArray(b)) {
 *     return a === b;
 *   }
 *   if (a.length !== b.length) {
 *     return false;
 *   }
 *   let tempB = [...b];
 *   function reduceToDetermineIfArraysContainSameContents(
 *     previousCallResult: boolean,
 *     arrayMember: any
 *   ): boolean {
 *     if (previousCallResult === false) {
 *       return false;
 *     }
 *     if (tempB.includes(arrayMember)) {
 *       tempB = removeMatch(tempB, arrayMember);
 *       return true;
 *     }
 *     return false;
 *   }
 *   return a.reduce(reduceToDetermineIfArraysContainSameContents, true);
 * }
 *
 * export const creactOrderDoesNotMatterSelector = createSelectorFactory(
 *   (projectionFun) => defaultMemoize(
 *     projectionFun,
 *     orderDoesNotMatterComparer,
 *     orderDoesNotMatterComparer
 *   )
 * );
 * ```
 *
 * **Creating an Alternative Memoization Strategy**
 *
 * ```ts
 * function serialize(x: any): string {
 *   return JSON.stringify(x);
 * }
 *
 * export const createFullHistorySelector = createSelectorFactory(
 *  (projectionFunction) => {
 *    const cache = {};
 *
 *    function memoized() {
 *      const serializedArguments = serialize(...arguments);
 *       if (cache[serializedArguments] != null) {
 *         cache[serializedArguments] = projectionFunction.apply(null, arguments);
 *       }
 *       return cache[serializedArguments];
 *     }
 *     return {
 *       memoized,
 *       reset: () => {},
 *       setResult: () => {},
 *       clearResult: () => {},
 *     };
 *   }
 * );
 * ```
 *
 *
 */
export function createSelectorFactory(memoize, options = {
    stateFn: defaultStateFn,
}) {
    return function (...input) {
        let args = input;
        if (Array.isArray(args[0])) {
            const [head, ...tail] = args;
            args = [...head, ...tail];
        }
        const selectors = args.slice(0, args.length - 1);
        const projector = args[args.length - 1];
        const memoizedSelectors = selectors.filter((selector) => selector.release && typeof selector.release === 'function');
        const memoizedProjector = memoize(function (...selectors) {
            return projector.apply(null, selectors);
        });
        const memoizedState = defaultMemoize(function (state, props) {
            return options.stateFn.apply(null, [
                state,
                selectors,
                props,
                memoizedProjector,
            ]);
        });
        function release() {
            memoizedState.reset();
            memoizedProjector.reset();
            memoizedSelectors.forEach((selector) => selector.release());
        }
        return Object.assign(memoizedState.memoized, {
            release,
            projector: memoizedProjector.memoized,
            setResult: memoizedState.setResult,
            clearResult: memoizedState.clearResult,
        });
    };
}
export function createFeatureSelector(featureName) {
    return createSelector((state) => {
        const featureState = state[featureName];
        if (!isNgrxMockEnvironment() && isDevMode() && !(featureName in state)) {
            console.warn(`@ngrx/store: The feature name "${featureName}" does ` +
                'not exist in the state, therefore createFeatureSelector ' +
                'cannot access it.  Be sure it is imported in a loaded module ' +
                `using StoreModule.forRoot('${featureName}', ...) or ` +
                `StoreModule.forFeature('${featureName}', ...).  If the default ` +
                'state is intended to be undefined, as is the case with router ' +
                'state, this development-only warning message can be ignored.');
        }
        return featureState;
    }, (featureState) => featureState);
}
//# sourceMappingURL=data:application/json;base64,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