import { isDevMode } from '@angular/core';
import { serializationCheckMetaReducer, immutabilityCheckMetaReducer, inNgZoneAssertMetaReducer, } from './meta-reducers';
import { _USER_RUNTIME_CHECKS, ACTIVE_RUNTIME_CHECKS, META_REDUCERS, USER_RUNTIME_CHECKS, _ACTION_TYPE_UNIQUENESS_CHECK, } from './tokens';
import { REGISTERED_ACTION_TYPES } from './globals';
import { RUNTIME_CHECK_URL } from './meta-reducers/utils';
export function createActiveRuntimeChecks(runtimeChecks) {
    if (isDevMode()) {
        return {
            strictStateSerializability: false,
            strictActionSerializability: false,
            strictStateImmutability: true,
            strictActionImmutability: true,
            strictActionWithinNgZone: false,
            strictActionTypeUniqueness: false,
            ...runtimeChecks,
        };
    }
    return {
        strictStateSerializability: false,
        strictActionSerializability: false,
        strictStateImmutability: false,
        strictActionImmutability: false,
        strictActionWithinNgZone: false,
        strictActionTypeUniqueness: false,
    };
}
export function createSerializationCheckMetaReducer({ strictActionSerializability, strictStateSerializability, }) {
    return (reducer) => strictActionSerializability || strictStateSerializability
        ? serializationCheckMetaReducer(reducer, {
            action: (action) => strictActionSerializability && !ignoreNgrxAction(action),
            state: () => strictStateSerializability,
        })
        : reducer;
}
export function createImmutabilityCheckMetaReducer({ strictActionImmutability, strictStateImmutability, }) {
    return (reducer) => strictActionImmutability || strictStateImmutability
        ? immutabilityCheckMetaReducer(reducer, {
            action: (action) => strictActionImmutability && !ignoreNgrxAction(action),
            state: () => strictStateImmutability,
        })
        : reducer;
}
function ignoreNgrxAction(action) {
    return action.type.startsWith('@ngrx');
}
export function createInNgZoneCheckMetaReducer({ strictActionWithinNgZone, }) {
    return (reducer) => strictActionWithinNgZone
        ? inNgZoneAssertMetaReducer(reducer, {
            action: (action) => strictActionWithinNgZone && !ignoreNgrxAction(action),
        })
        : reducer;
}
export function provideRuntimeChecks(runtimeChecks) {
    return [
        {
            provide: _USER_RUNTIME_CHECKS,
            useValue: runtimeChecks,
        },
        {
            provide: USER_RUNTIME_CHECKS,
            useFactory: _runtimeChecksFactory,
            deps: [_USER_RUNTIME_CHECKS],
        },
        {
            provide: ACTIVE_RUNTIME_CHECKS,
            deps: [USER_RUNTIME_CHECKS],
            useFactory: createActiveRuntimeChecks,
        },
        {
            provide: META_REDUCERS,
            multi: true,
            deps: [ACTIVE_RUNTIME_CHECKS],
            useFactory: createImmutabilityCheckMetaReducer,
        },
        {
            provide: META_REDUCERS,
            multi: true,
            deps: [ACTIVE_RUNTIME_CHECKS],
            useFactory: createSerializationCheckMetaReducer,
        },
        {
            provide: META_REDUCERS,
            multi: true,
            deps: [ACTIVE_RUNTIME_CHECKS],
            useFactory: createInNgZoneCheckMetaReducer,
        },
    ];
}
export function checkForActionTypeUniqueness() {
    return [
        {
            provide: _ACTION_TYPE_UNIQUENESS_CHECK,
            multi: true,
            deps: [ACTIVE_RUNTIME_CHECKS],
            useFactory: _actionTypeUniquenessCheck,
        },
    ];
}
export function _runtimeChecksFactory(runtimeChecks) {
    return runtimeChecks;
}
export function _actionTypeUniquenessCheck(config) {
    if (!config.strictActionTypeUniqueness) {
        return;
    }
    const duplicates = Object.entries(REGISTERED_ACTION_TYPES)
        .filter(([, registrations]) => registrations > 1)
        .map(([type]) => type);
    if (duplicates.length) {
        throw new Error(`Action types are registered more than once, ${duplicates
            .map((type) => `"${type}"`)
            .join(', ')}. ${RUNTIME_CHECK_URL}#strictactiontypeuniqueness`);
    }
}
//# sourceMappingURL=data:application/json;base64,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