import { capitalize } from './helpers';
import { isPlainObject } from './meta-reducers/utils';
import { createFeatureSelector, createSelector, } from './selector';
/**
 * @description
 * A function that accepts a feature name and a feature reducer, and creates
 * a feature selector and a selector for each feature state property.
 *
 * @param featureConfig An object that contains a feature name and a feature reducer.
 * @returns An object that contains a feature name, a feature reducer,
 * a feature selector, and a selector for each feature state property.
 *
 * @usageNotes
 *
 * **With Application State**
 *
 * ```ts
 * interface AppState {
 *   products: ProductsState;
 * }
 *
 * interface ProductsState {
 *   products: Product[];
 *   selectedId: string | null;
 * }
 *
 * const initialState: ProductsState = {
 *   products: [],
 *   selectedId: null,
 * };
 *
 * // AppState is passed as a generic argument
 * const productsFeature = createFeature<AppState>({
 *   name: 'products',
 *   reducer: createReducer(
 *     initialState,
 *     on(ProductsApiActions.loadSuccess(state, { products }) => ({
 *       ...state,
 *       products,
 *     }),
 *   ),
 * });
 *
 * const {
 *   selectProductsState, // type: MemoizedSelector<AppState, ProductsState>
 *   selectProducts, // type: MemoizedSelector<AppState, Product[]>
 *   selectSelectedId, // type: MemoizedSelector<AppState, string | null>
 * } = productsFeature;
 * ```
 *
 * **Without Application State**
 *
 * ```ts
 * const productsFeature = createFeature({
 *   name: 'products',
 *   reducer: createReducer(initialState),
 * });
 *
 * const {
 *   selectProductsState, // type: MemoizedSelector<Record<string, any>, ProductsState>
 *   selectProducts, // type: MemoizedSelector<Record<string, any>, Product[]>
 *   selectSelectedId, // type: MemoizedSelector<Record<string, any, string | null>
 * } = productsFeature;
 * ```
 */
export function createFeature(featureConfig) {
    const { name, reducer } = featureConfig;
    const featureSelector = createFeatureSelector(name);
    const nestedSelectors = createNestedSelectors(featureSelector, reducer);
    return {
        name,
        reducer,
        [`select${capitalize(name)}State`]: featureSelector,
        ...nestedSelectors,
    };
}
function createNestedSelectors(featureSelector, reducer) {
    const initialState = getInitialState(reducer);
    const nestedKeys = (isPlainObject(initialState)
        ? Object.keys(initialState)
        : []);
    return nestedKeys.reduce((nestedSelectors, nestedKey) => ({
        ...nestedSelectors,
        [`select${capitalize(nestedKey)}`]: createSelector(featureSelector, (parentState) => parentState?.[nestedKey]),
    }), {});
}
function getInitialState(reducer) {
    return reducer(undefined, { type: '@ngrx/feature/init' });
}
//# sourceMappingURL=data:application/json;base64,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