import * as Actions from './actions';
export function difference(first, second) {
    return first.filter((item) => second.indexOf(item) < 0);
}
/**
 * Provides an app's view into the state of the lifted store.
 */
export function unliftState(liftedState) {
    const { computedStates, currentStateIndex } = liftedState;
    // At start up NgRx dispatches init actions,
    // When these init actions are being filtered out by the predicate or safe/block list options
    // we don't have a complete computed states yet.
    // At this point it could happen that we're out of bounds, when this happens we fall back to the last known state
    if (currentStateIndex >= computedStates.length) {
        const { state } = computedStates[computedStates.length - 1];
        return state;
    }
    const { state } = computedStates[currentStateIndex];
    return state;
}
export function unliftAction(liftedState) {
    return liftedState.actionsById[liftedState.nextActionId - 1];
}
/**
 * Lifts an app's action into an action on the lifted store.
 */
export function liftAction(action) {
    return new Actions.PerformAction(action, +Date.now());
}
/**
 * Sanitizes given actions with given function.
 */
export function sanitizeActions(actionSanitizer, actions) {
    return Object.keys(actions).reduce((sanitizedActions, actionIdx) => {
        const idx = Number(actionIdx);
        sanitizedActions[idx] = sanitizeAction(actionSanitizer, actions[idx], idx);
        return sanitizedActions;
    }, {});
}
/**
 * Sanitizes given action with given function.
 */
export function sanitizeAction(actionSanitizer, action, actionIdx) {
    return {
        ...action,
        action: actionSanitizer(action.action, actionIdx),
    };
}
/**
 * Sanitizes given states with given function.
 */
export function sanitizeStates(stateSanitizer, states) {
    return states.map((computedState, idx) => ({
        state: sanitizeState(stateSanitizer, computedState.state, idx),
        error: computedState.error,
    }));
}
/**
 * Sanitizes given state with given function.
 */
export function sanitizeState(stateSanitizer, state, stateIdx) {
    return stateSanitizer(state, stateIdx);
}
/**
 * Read the config and tell if actions should be filtered
 */
export function shouldFilterActions(config) {
    return config.predicate || config.actionsSafelist || config.actionsBlocklist;
}
/**
 * Return a full filtered lifted state
 */
export function filterLiftedState(liftedState, predicate, safelist, blocklist) {
    const filteredStagedActionIds = [];
    const filteredActionsById = {};
    const filteredComputedStates = [];
    liftedState.stagedActionIds.forEach((id, idx) => {
        const liftedAction = liftedState.actionsById[id];
        if (!liftedAction)
            return;
        if (idx &&
            isActionFiltered(liftedState.computedStates[idx], liftedAction, predicate, safelist, blocklist)) {
            return;
        }
        filteredActionsById[id] = liftedAction;
        filteredStagedActionIds.push(id);
        filteredComputedStates.push(liftedState.computedStates[idx]);
    });
    return {
        ...liftedState,
        stagedActionIds: filteredStagedActionIds,
        actionsById: filteredActionsById,
        computedStates: filteredComputedStates,
    };
}
/**
 * Return true is the action should be ignored
 */
export function isActionFiltered(state, action, predicate, safelist, blockedlist) {
    const predicateMatch = predicate && !predicate(state, action.action);
    const safelistMatch = safelist &&
        !action.action.type.match(safelist.map((s) => escapeRegExp(s)).join('|'));
    const blocklistMatch = blockedlist &&
        action.action.type.match(blockedlist.map((s) => escapeRegExp(s)).join('|'));
    return predicateMatch || safelistMatch || blocklistMatch;
}
/**
 * Return string with escaped RegExp special characters
 * https://stackoverflow.com/a/6969486/1337347
 */
function escapeRegExp(s) {
    return s.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
}
//# sourceMappingURL=data:application/json;base64,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