import { Inject, Injectable, InjectionToken } from '@angular/core';
import { UPDATE } from '@ngrx/store';
import { EMPTY, Observable, of } from 'rxjs';
import { catchError, concatMap, debounceTime, filter, map, share, switchMap, take, takeUntil, timeout, } from 'rxjs/operators';
import { IMPORT_STATE, PERFORM_ACTION } from './actions';
import { STORE_DEVTOOLS_CONFIG, } from './config';
import { isActionFiltered, sanitizeAction, sanitizeActions, sanitizeState, sanitizeStates, shouldFilterActions, unliftState, } from './utils';
import * as i0 from "@angular/core";
import * as i1 from "./devtools-dispatcher";
import * as i2 from "./config";
export const ExtensionActionTypes = {
    START: 'START',
    DISPATCH: 'DISPATCH',
    STOP: 'STOP',
    ACTION: 'ACTION',
};
export const REDUX_DEVTOOLS_EXTENSION = new InjectionToken('@ngrx/store-devtools Redux Devtools Extension');
export class DevtoolsExtension {
    constructor(devtoolsExtension, config, dispatcher) {
        this.config = config;
        this.dispatcher = dispatcher;
        this.devtoolsExtension = devtoolsExtension;
        this.createActionStreams();
    }
    notify(action, state) {
        if (!this.devtoolsExtension) {
            return;
        }
        // Check to see if the action requires a full update of the liftedState.
        // If it is a simple action generated by the user's app and the recording
        // is not locked/paused, only send the action and the current state (fast).
        //
        // A full liftedState update (slow: serializes the entire liftedState) is
        // only required when:
        //   a) redux-devtools-extension fires the @@Init action (ignored by
        //      @ngrx/store-devtools)
        //   b) an action is generated by an @ngrx module (e.g. @ngrx/effects/init
        //      or @ngrx/store/update-reducers)
        //   c) the state has been recomputed due to time-traveling
        //   d) any action that is not a PerformAction to err on the side of
        //      caution.
        if (action.type === PERFORM_ACTION) {
            if (state.isLocked || state.isPaused) {
                return;
            }
            const currentState = unliftState(state);
            if (shouldFilterActions(this.config) &&
                isActionFiltered(currentState, action, this.config.predicate, this.config.actionsSafelist, this.config.actionsBlocklist)) {
                return;
            }
            const sanitizedState = this.config.stateSanitizer
                ? sanitizeState(this.config.stateSanitizer, currentState, state.currentStateIndex)
                : currentState;
            const sanitizedAction = this.config.actionSanitizer
                ? sanitizeAction(this.config.actionSanitizer, action, state.nextActionId)
                : action;
            this.sendToReduxDevtools(() => this.extensionConnection.send(sanitizedAction, sanitizedState));
        }
        else {
            // Requires full state update
            const sanitizedLiftedState = {
                ...state,
                stagedActionIds: state.stagedActionIds,
                actionsById: this.config.actionSanitizer
                    ? sanitizeActions(this.config.actionSanitizer, state.actionsById)
                    : state.actionsById,
                computedStates: this.config.stateSanitizer
                    ? sanitizeStates(this.config.stateSanitizer, state.computedStates)
                    : state.computedStates,
            };
            this.sendToReduxDevtools(() => this.devtoolsExtension.send(null, sanitizedLiftedState, this.getExtensionConfig(this.config)));
        }
    }
    createChangesObservable() {
        if (!this.devtoolsExtension) {
            return EMPTY;
        }
        return new Observable((subscriber) => {
            const connection = this.devtoolsExtension.connect(this.getExtensionConfig(this.config));
            this.extensionConnection = connection;
            connection.init();
            connection.subscribe((change) => subscriber.next(change));
            return connection.unsubscribe;
        });
    }
    createActionStreams() {
        // Listens to all changes
        const changes$ = this.createChangesObservable().pipe(share());
        // Listen for the start action
        const start$ = changes$.pipe(filter((change) => change.type === ExtensionActionTypes.START));
        // Listen for the stop action
        const stop$ = changes$.pipe(filter((change) => change.type === ExtensionActionTypes.STOP));
        // Listen for lifted actions
        const liftedActions$ = changes$.pipe(filter((change) => change.type === ExtensionActionTypes.DISPATCH), map((change) => this.unwrapAction(change.payload)), concatMap((action) => {
            if (action.type === IMPORT_STATE) {
                // State imports may happen in two situations:
                // 1. Explicitly by user
                // 2. User activated the "persist state accross reloads" option
                //    and now the state is imported during reload.
                // Because of option 2, we need to give possible
                // lazy loaded reducers time to instantiate.
                // As soon as there is no UPDATE action within 1 second,
                // it is assumed that all reducers are loaded.
                return this.dispatcher.pipe(filter((action) => action.type === UPDATE), timeout(1000), debounceTime(1000), map(() => action), catchError(() => of(action)), take(1));
            }
            else {
                return of(action);
            }
        }));
        // Listen for unlifted actions
        const actions$ = changes$.pipe(filter((change) => change.type === ExtensionActionTypes.ACTION), map((change) => this.unwrapAction(change.payload)));
        const actionsUntilStop$ = actions$.pipe(takeUntil(stop$));
        const liftedUntilStop$ = liftedActions$.pipe(takeUntil(stop$));
        this.start$ = start$.pipe(takeUntil(stop$));
        // Only take the action sources between the start/stop events
        this.actions$ = this.start$.pipe(switchMap(() => actionsUntilStop$));
        this.liftedActions$ = this.start$.pipe(switchMap(() => liftedUntilStop$));
    }
    unwrapAction(action) {
        return typeof action === 'string' ? eval(`(${action})`) : action;
    }
    getExtensionConfig(config) {
        const extensionOptions = {
            name: config.name,
            features: config.features,
            serialize: config.serialize,
            autoPause: config.autoPause ?? false,
            // The action/state sanitizers are not added to the config
            // because sanitation is done in this class already.
            // It is done before sending it to the devtools extension for consistency:
            // - If we call extensionConnection.send(...),
            //   the extension would call the sanitizers.
            // - If we call devtoolsExtension.send(...) (aka full state update),
            //   the extension would NOT call the sanitizers, so we have to do it ourselves.
        };
        if (config.maxAge !== false /* support === 0 */) {
            extensionOptions.maxAge = config.maxAge;
        }
        return extensionOptions;
    }
    sendToReduxDevtools(send) {
        try {
            send();
        }
        catch (err) {
            console.warn('@ngrx/store-devtools: something went wrong inside the redux devtools', err);
        }
    }
}
/** @nocollapse */ /** @nocollapse */ DevtoolsExtension.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "13.0.0", ngImport: i0, type: DevtoolsExtension, deps: [{ token: REDUX_DEVTOOLS_EXTENSION }, { token: STORE_DEVTOOLS_CONFIG }, { token: i1.DevtoolsDispatcher }], target: i0.ɵɵFactoryTarget.Injectable });
/** @nocollapse */ /** @nocollapse */ DevtoolsExtension.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "13.0.0", ngImport: i0, type: DevtoolsExtension });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "13.0.0", ngImport: i0, type: DevtoolsExtension, decorators: [{
            type: Injectable
        }], ctorParameters: function () { return [{ type: undefined, decorators: [{
                    type: Inject,
                    args: [REDUX_DEVTOOLS_EXTENSION]
                }] }, { type: i2.StoreDevtoolsConfig, decorators: [{
                    type: Inject,
                    args: [STORE_DEVTOOLS_CONFIG]
                }] }, { type: i1.DevtoolsDispatcher }]; } });
//# sourceMappingURL=data:application/json;base64,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