import { of, pipe } from 'rxjs';
import { concatMap, withLatestFrom } from 'rxjs/operators';
/**
 * `concatLatestFrom` combines the source value
 * and the last available value from a lazily evaluated Observable
 * in a new array
 *
 * @usageNotes
 *
 * Select the active customer from the NgRx Store
 *
 * ```ts
 * import { concatLatestFrom } from '@ngrx/effects';
 * import * as fromCustomers from '../customers';
 *
 * this.actions$.pipe(
 *  concatLatestFrom(() => this.store.select(fromCustomers.selectActiveCustomer))
 * )
 * ```
 *
 * Select a customer from the NgRx Store by its id that is available on the action
 *
 * ```ts
 * import { concatLatestFrom } from '@ngrx/effects';
 * import * fromCustomers from '../customers';
 *
 * this.actions$.pipe(
 *  concatLatestFrom((action) => this.store.select(fromCustomers.selectCustomer(action.customerId)))
 * )
 * ```
 */
export function concatLatestFrom(observablesFactory) {
    return pipe(concatMap((value) => {
        const observables = observablesFactory(value);
        const observablesAsArray = Array.isArray(observables)
            ? observables
            : [observables];
        return of(value).pipe(withLatestFrom(...observablesAsArray));
    }));
}
//# sourceMappingURL=data:application/json;base64,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