import { defer, merge, Notification, Subject, } from 'rxjs';
import { concatMap, dematerialize, filter, finalize, map, materialize, } from 'rxjs/operators';
/**
 * Wraps project fn with error handling making it safe to use in Effects.
 * Takes either a config with named properties that represent different possible
 * callbacks or project/error callbacks that are required.
 */
export function act(
/** Allow to take either config object or project/error functions */
configOrProject, errorFn) {
    const { project, error, complete, operator, unsubscribe } = typeof configOrProject === 'function'
        ? {
            project: configOrProject,
            // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
            error: errorFn,
            operator: concatMap,
            complete: undefined,
            unsubscribe: undefined,
        }
        : { ...configOrProject, operator: configOrProject.operator || concatMap };
    return (source) => defer(() => {
        const subject = new Subject();
        return merge(source.pipe(operator((input, index) => defer(() => {
            let completed = false;
            let errored = false;
            let projectedCount = 0;
            return project(input, index).pipe(materialize(), map((notification) => {
                switch (notification.kind) {
                    case 'E':
                        errored = true;
                        return new Notification('N', error(notification.error, input));
                    case 'C':
                        completed = true;
                        return complete
                            ? new Notification('N', complete(projectedCount, input))
                            : undefined;
                    default:
                        ++projectedCount;
                        return notification;
                }
            }), filter((n) => n != null), dematerialize(), finalize(() => {
                if (!completed && !errored && unsubscribe) {
                    subject.next(unsubscribe(projectedCount, input));
                }
            }));
        }))), subject);
    });
}
//# sourceMappingURL=data:application/json;base64,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