/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
// Below are constants for LView indices to help us look up LView members
// without having to remember the specific indices.
// Uglify will inline these when minifying so there shouldn't be a cost.
export const HOST = 0;
export const TVIEW = 1;
export const FLAGS = 2;
export const PARENT = 3;
export const NEXT = 4;
export const TRANSPLANTED_VIEWS_TO_REFRESH = 5;
export const T_HOST = 6;
export const CLEANUP = 7;
export const CONTEXT = 8;
export const INJECTOR = 9;
export const RENDERER_FACTORY = 10;
export const RENDERER = 11;
export const SANITIZER = 12;
export const CHILD_HEAD = 13;
export const CHILD_TAIL = 14;
// FIXME(misko): Investigate if the three declarations aren't all same thing.
export const DECLARATION_VIEW = 15;
export const DECLARATION_COMPONENT_VIEW = 16;
export const DECLARATION_LCONTAINER = 17;
export const PREORDER_HOOK_FLAGS = 18;
export const QUERIES = 19;
/**
 * Size of LView's header. Necessary to adjust for it when setting slots.
 *
 * IMPORTANT: `HEADER_OFFSET` should only be referred to the in the `ɵɵ*` instructions to translate
 * instruction index into `LView` index. All other indexes should be in the `LView` index space and
 * there should be no need to refer to `HEADER_OFFSET` anywhere else.
 */
export const HEADER_OFFSET = 20;
/**
 * Converts `TViewType` into human readable text.
 * Make sure this matches with `TViewType`
 */
export const TViewTypeAsString = [
    'Root',
    'Component',
    'Embedded', // 2
];
// Note: This hack is necessary so we don't erroneously get a circular dependency
// failure based on types.
export const unusedValueExportToPlacateAjd = 1;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidmlldy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uLy4uLy4uLy4uLy4uL3BhY2thZ2VzL2NvcmUvc3JjL3JlbmRlcjMvaW50ZXJmYWNlcy92aWV3LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFBOzs7Ozs7R0FNRztBQW9CSCx5RUFBeUU7QUFDekUsbURBQW1EO0FBQ25ELHdFQUF3RTtBQUN4RSxNQUFNLENBQUMsTUFBTSxJQUFJLEdBQUcsQ0FBQyxDQUFDO0FBQ3RCLE1BQU0sQ0FBQyxNQUFNLEtBQUssR0FBRyxDQUFDLENBQUM7QUFDdkIsTUFBTSxDQUFDLE1BQU0sS0FBSyxHQUFHLENBQUMsQ0FBQztBQUN2QixNQUFNLENBQUMsTUFBTSxNQUFNLEdBQUcsQ0FBQyxDQUFDO0FBQ3hCLE1BQU0sQ0FBQyxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7QUFDdEIsTUFBTSxDQUFDLE1BQU0sNkJBQTZCLEdBQUcsQ0FBQyxDQUFDO0FBQy9DLE1BQU0sQ0FBQyxNQUFNLE1BQU0sR0FBRyxDQUFDLENBQUM7QUFDeEIsTUFBTSxDQUFDLE1BQU0sT0FBTyxHQUFHLENBQUMsQ0FBQztBQUN6QixNQUFNLENBQUMsTUFBTSxPQUFPLEdBQUcsQ0FBQyxDQUFDO0FBQ3pCLE1BQU0sQ0FBQyxNQUFNLFFBQVEsR0FBRyxDQUFDLENBQUM7QUFDMUIsTUFBTSxDQUFDLE1BQU0sZ0JBQWdCLEdBQUcsRUFBRSxDQUFDO0FBQ25DLE1BQU0sQ0FBQyxNQUFNLFFBQVEsR0FBRyxFQUFFLENBQUM7QUFDM0IsTUFBTSxDQUFDLE1BQU0sU0FBUyxHQUFHLEVBQUUsQ0FBQztBQUM1QixNQUFNLENBQUMsTUFBTSxVQUFVLEdBQUcsRUFBRSxDQUFDO0FBQzdCLE1BQU0sQ0FBQyxNQUFNLFVBQVUsR0FBRyxFQUFFLENBQUM7QUFDN0IsNkVBQTZFO0FBQzdFLE1BQU0sQ0FBQyxNQUFNLGdCQUFnQixHQUFHLEVBQUUsQ0FBQztBQUNuQyxNQUFNLENBQUMsTUFBTSwwQkFBMEIsR0FBRyxFQUFFLENBQUM7QUFDN0MsTUFBTSxDQUFDLE1BQU0sc0JBQXNCLEdBQUcsRUFBRSxDQUFDO0FBQ3pDLE1BQU0sQ0FBQyxNQUFNLG1CQUFtQixHQUFHLEVBQUUsQ0FBQztBQUN0QyxNQUFNLENBQUMsTUFBTSxPQUFPLEdBQUcsRUFBRSxDQUFDO0FBQzFCOzs7Ozs7R0FNRztBQUNILE1BQU0sQ0FBQyxNQUFNLGFBQWEsR0FBRyxFQUFFLENBQUM7QUFxZGhDOzs7R0FHRztBQUNILE1BQU0sQ0FBQyxNQUFNLGlCQUFpQixHQUFHO0lBQy9CLE1BQU07SUFDTixXQUFXO0lBQ1gsVUFBVSxFQUFJLElBQUk7Q0FDVixDQUFDO0FBd1hYLGlGQUFpRjtBQUNqRiwwQkFBMEI7QUFDMUIsTUFBTSxDQUFDLE1BQU0sNkJBQTZCLEdBQUcsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBAbGljZW5zZVxuICogQ29weXJpZ2h0IEdvb2dsZSBMTEMgQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqXG4gKiBVc2Ugb2YgdGhpcyBzb3VyY2UgY29kZSBpcyBnb3Zlcm5lZCBieSBhbiBNSVQtc3R5bGUgbGljZW5zZSB0aGF0IGNhbiBiZVxuICogZm91bmQgaW4gdGhlIExJQ0VOU0UgZmlsZSBhdCBodHRwczovL2FuZ3VsYXIuaW8vbGljZW5zZVxuICovXG5cbmltcG9ydCB7SW5qZWN0b3J9IGZyb20gJy4uLy4uL2RpL2luamVjdG9yJztcbmltcG9ydCB7UHJvdmlkZXJUb2tlbn0gZnJvbSAnLi4vLi4vZGkvcHJvdmlkZXJfdG9rZW4nO1xuaW1wb3J0IHtUeXBlfSBmcm9tICcuLi8uLi9pbnRlcmZhY2UvdHlwZSc7XG5pbXBvcnQge1NjaGVtYU1ldGFkYXRhfSBmcm9tICcuLi8uLi9tZXRhZGF0YS9zY2hlbWEnO1xuaW1wb3J0IHtTYW5pdGl6ZXJ9IGZyb20gJy4uLy4uL3Nhbml0aXphdGlvbi9zYW5pdGl6ZXInO1xuXG5pbXBvcnQge0xDb250YWluZXJ9IGZyb20gJy4vY29udGFpbmVyJztcbmltcG9ydCB7Q29tcG9uZW50RGVmLCBDb21wb25lbnRUZW1wbGF0ZSwgRGlyZWN0aXZlRGVmLCBEaXJlY3RpdmVEZWZMaXN0LCBIb3N0QmluZGluZ3NGdW5jdGlvbiwgUGlwZURlZiwgUGlwZURlZkxpc3QsIFZpZXdRdWVyaWVzRnVuY3Rpb259IGZyb20gJy4vZGVmaW5pdGlvbic7XG5pbXBvcnQge0kxOG5VcGRhdGVPcENvZGVzLCBUSTE4biwgVEljdX0gZnJvbSAnLi9pMThuJztcbmltcG9ydCB7VENvbnN0YW50cywgVE5vZGV9IGZyb20gJy4vbm9kZSc7XG5pbXBvcnQge1BsYXllckhhbmRsZXJ9IGZyb20gJy4vcGxheWVyJztcbmltcG9ydCB7TFF1ZXJpZXMsIFRRdWVyaWVzfSBmcm9tICcuL3F1ZXJ5JztcbmltcG9ydCB7UmVuZGVyZXIzLCBSZW5kZXJlckZhY3RvcnkzfSBmcm9tICcuL3JlbmRlcmVyJztcbmltcG9ydCB7UkNvbW1lbnQsIFJFbGVtZW50fSBmcm9tICcuL3JlbmRlcmVyX2RvbSc7XG5pbXBvcnQge1RTdHlsaW5nS2V5LCBUU3R5bGluZ1JhbmdlfSBmcm9tICcuL3N0eWxpbmcnO1xuXG5cblxuLy8gQmVsb3cgYXJlIGNvbnN0YW50cyBmb3IgTFZpZXcgaW5kaWNlcyB0byBoZWxwIHVzIGxvb2sgdXAgTFZpZXcgbWVtYmVyc1xuLy8gd2l0aG91dCBoYXZpbmcgdG8gcmVtZW1iZXIgdGhlIHNwZWNpZmljIGluZGljZXMuXG4vLyBVZ2xpZnkgd2lsbCBpbmxpbmUgdGhlc2Ugd2hlbiBtaW5pZnlpbmcgc28gdGhlcmUgc2hvdWxkbid0IGJlIGEgY29zdC5cbmV4cG9ydCBjb25zdCBIT1NUID0gMDtcbmV4cG9ydCBjb25zdCBUVklFVyA9IDE7XG5leHBvcnQgY29uc3QgRkxBR1MgPSAyO1xuZXhwb3J0IGNvbnN0IFBBUkVOVCA9IDM7XG5leHBvcnQgY29uc3QgTkVYVCA9IDQ7XG5leHBvcnQgY29uc3QgVFJBTlNQTEFOVEVEX1ZJRVdTX1RPX1JFRlJFU0ggPSA1O1xuZXhwb3J0IGNvbnN0IFRfSE9TVCA9IDY7XG5leHBvcnQgY29uc3QgQ0xFQU5VUCA9IDc7XG5leHBvcnQgY29uc3QgQ09OVEVYVCA9IDg7XG5leHBvcnQgY29uc3QgSU5KRUNUT1IgPSA5O1xuZXhwb3J0IGNvbnN0IFJFTkRFUkVSX0ZBQ1RPUlkgPSAxMDtcbmV4cG9ydCBjb25zdCBSRU5ERVJFUiA9IDExO1xuZXhwb3J0IGNvbnN0IFNBTklUSVpFUiA9IDEyO1xuZXhwb3J0IGNvbnN0IENISUxEX0hFQUQgPSAxMztcbmV4cG9ydCBjb25zdCBDSElMRF9UQUlMID0gMTQ7XG4vLyBGSVhNRShtaXNrbyk6IEludmVzdGlnYXRlIGlmIHRoZSB0aHJlZSBkZWNsYXJhdGlvbnMgYXJlbid0IGFsbCBzYW1lIHRoaW5nLlxuZXhwb3J0IGNvbnN0IERFQ0xBUkFUSU9OX1ZJRVcgPSAxNTtcbmV4cG9ydCBjb25zdCBERUNMQVJBVElPTl9DT01QT05FTlRfVklFVyA9IDE2O1xuZXhwb3J0IGNvbnN0IERFQ0xBUkFUSU9OX0xDT05UQUlORVIgPSAxNztcbmV4cG9ydCBjb25zdCBQUkVPUkRFUl9IT09LX0ZMQUdTID0gMTg7XG5leHBvcnQgY29uc3QgUVVFUklFUyA9IDE5O1xuLyoqXG4gKiBTaXplIG9mIExWaWV3J3MgaGVhZGVyLiBOZWNlc3NhcnkgdG8gYWRqdXN0IGZvciBpdCB3aGVuIHNldHRpbmcgc2xvdHMuXG4gKlxuICogSU1QT1JUQU5UOiBgSEVBREVSX09GRlNFVGAgc2hvdWxkIG9ubHkgYmUgcmVmZXJyZWQgdG8gdGhlIGluIHRoZSBgybXJtSpgIGluc3RydWN0aW9ucyB0byB0cmFuc2xhdGVcbiAqIGluc3RydWN0aW9uIGluZGV4IGludG8gYExWaWV3YCBpbmRleC4gQWxsIG90aGVyIGluZGV4ZXMgc2hvdWxkIGJlIGluIHRoZSBgTFZpZXdgIGluZGV4IHNwYWNlIGFuZFxuICogdGhlcmUgc2hvdWxkIGJlIG5vIG5lZWQgdG8gcmVmZXIgdG8gYEhFQURFUl9PRkZTRVRgIGFueXdoZXJlIGVsc2UuXG4gKi9cbmV4cG9ydCBjb25zdCBIRUFERVJfT0ZGU0VUID0gMjA7XG5cblxuLy8gVGhpcyBpbnRlcmZhY2UgcmVwbGFjZXMgdGhlIHJlYWwgTFZpZXcgaW50ZXJmYWNlIGlmIGl0IGlzIGFuIGFyZyBvciBhXG4vLyByZXR1cm4gdmFsdWUgb2YgYSBwdWJsaWMgaW5zdHJ1Y3Rpb24uIFRoaXMgZW5zdXJlcyB3ZSBkb24ndCBuZWVkIHRvIGV4cG9zZVxuLy8gdGhlIGFjdHVhbCBpbnRlcmZhY2UsIHdoaWNoIHNob3VsZCBiZSBrZXB0IHByaXZhdGUuXG5leHBvcnQgaW50ZXJmYWNlIE9wYXF1ZVZpZXdTdGF0ZSB7XG4gICdfX2JyYW5kX18nOiAnQnJhbmQgZm9yIE9wYXF1ZVZpZXdTdGF0ZSB0aGF0IG5vdGhpbmcgd2lsbCBtYXRjaCc7XG59XG5cblxuLyoqXG4gKiBgTFZpZXdgIHN0b3JlcyBhbGwgb2YgdGhlIGluZm9ybWF0aW9uIG5lZWRlZCB0byBwcm9jZXNzIHRoZSBpbnN0cnVjdGlvbnMgYXNcbiAqIHRoZXkgYXJlIGludm9rZWQgZnJvbSB0aGUgdGVtcGxhdGUuIEVhY2ggZW1iZWRkZWQgdmlldyBhbmQgY29tcG9uZW50IHZpZXcgaGFzIGl0c1xuICogb3duIGBMVmlld2AuIFdoZW4gcHJvY2Vzc2luZyBhIHBhcnRpY3VsYXIgdmlldywgd2Ugc2V0IHRoZSBgdmlld0RhdGFgIHRvIHRoYXRcbiAqIGBMVmlld2AuIFdoZW4gdGhhdCB2aWV3IGlzIGRvbmUgcHJvY2Vzc2luZywgdGhlIGB2aWV3RGF0YWAgaXMgc2V0IGJhY2sgdG9cbiAqIHdoYXRldmVyIHRoZSBvcmlnaW5hbCBgdmlld0RhdGFgIHdhcyBiZWZvcmUgKHRoZSBwYXJlbnQgYExWaWV3YCkuXG4gKlxuICogS2VlcGluZyBzZXBhcmF0ZSBzdGF0ZSBmb3IgZWFjaCB2aWV3IGZhY2lsaXRpZXMgdmlldyBpbnNlcnRpb24gLyBkZWxldGlvbiwgc28gd2VcbiAqIGRvbid0IGhhdmUgdG8gZWRpdCB0aGUgZGF0YSBhcnJheSBiYXNlZCBvbiB3aGljaCB2aWV3cyBhcmUgcHJlc2VudC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBMVmlldyBleHRlbmRzIEFycmF5PGFueT4ge1xuICAvKipcbiAgICogSHVtYW4gcmVhZGFibGUgcmVwcmVzZW50YXRpb24gb2YgdGhlIGBMVmlld2AuXG4gICAqXG4gICAqIE5PVEU6IFRoaXMgcHJvcGVydHkgb25seSBleGlzdHMgaWYgYG5nRGV2TW9kZWAgaXMgc2V0IHRvIGB0cnVlYCBhbmQgaXQgaXMgbm90IHByZXNlbnQgaW5cbiAgICogcHJvZHVjdGlvbi4gSXRzIHByZXNlbmNlIGlzIHB1cmVseSB0byBoZWxwIGRlYnVnIGlzc3VlIGluIGRldmVsb3BtZW50LCBhbmQgc2hvdWxkIG5vdCBiZSByZWxpZWRcbiAgICogb24gaW4gcHJvZHVjdGlvbiBhcHBsaWNhdGlvbi5cbiAgICovXG4gIGRlYnVnPzogTFZpZXdEZWJ1ZztcblxuICAvKipcbiAgICogVGhlIG5vZGUgaW50byB3aGljaCB0aGlzIGBMVmlld2AgaXMgaW5zZXJ0ZWQuXG4gICAqL1xuICBbSE9TVF06IFJFbGVtZW50fG51bGw7XG5cbiAgLyoqXG4gICAqIFRoZSBzdGF0aWMgZGF0YSBmb3IgdGhpcyB2aWV3LiBXZSBuZWVkIGEgcmVmZXJlbmNlIHRvIHRoaXMgc28gd2UgY2FuIGVhc2lseSB3YWxrIHVwIHRoZVxuICAgKiBub2RlIHRyZWUgaW4gREkgYW5kIGdldCB0aGUgVFZpZXcuZGF0YSBhcnJheSBhc3NvY2lhdGVkIHdpdGggYSBub2RlICh3aGVyZSB0aGVcbiAgICogZGlyZWN0aXZlIGRlZnMgYXJlIHN0b3JlZCkuXG4gICAqL1xuICByZWFkb25seVtUVklFV106IFRWaWV3O1xuXG4gIC8qKiBGbGFncyBmb3IgdGhpcyB2aWV3LiBTZWUgTFZpZXdGbGFncyBmb3IgbW9yZSBpbmZvLiAqL1xuICBbRkxBR1NdOiBMVmlld0ZsYWdzO1xuXG4gIC8qKlxuICAgKiBUaGlzIG1heSBzdG9yZSBhbiB7QGxpbmsgTFZpZXd9IG9yIHtAbGluayBMQ29udGFpbmVyfS5cbiAgICpcbiAgICogYExWaWV3YCAtIFRoZSBwYXJlbnQgdmlldy4gVGhpcyBpcyBuZWVkZWQgd2hlbiB3ZSBleGl0IHRoZSB2aWV3IGFuZCBtdXN0IHJlc3RvcmUgdGhlIHByZXZpb3VzXG4gICAqIExWaWV3LiBXaXRob3V0IHRoaXMsIHRoZSByZW5kZXIgbWV0aG9kIHdvdWxkIGhhdmUgdG8ga2VlcCBhIHN0YWNrIG9mXG4gICAqIHZpZXdzIGFzIGl0IGlzIHJlY3Vyc2l2ZWx5IHJlbmRlcmluZyB0ZW1wbGF0ZXMuXG4gICAqXG4gICAqIGBMQ29udGFpbmVyYCAtIFRoZSBjdXJyZW50IHZpZXcgaXMgcGFydCBvZiBhIGNvbnRhaW5lciwgYW5kIGlzIGFuIGVtYmVkZGVkIHZpZXcuXG4gICAqL1xuICBbUEFSRU5UXTogTFZpZXd8TENvbnRhaW5lcnxudWxsO1xuXG4gIC8qKlxuICAgKlxuICAgKiBUaGUgbmV4dCBzaWJsaW5nIExWaWV3IG9yIExDb250YWluZXIuXG4gICAqXG4gICAqIEFsbG93cyB1cyB0byBwcm9wYWdhdGUgYmV0d2VlbiBzaWJsaW5nIHZpZXcgc3RhdGVzIHRoYXQgYXJlbid0IGluIHRoZSBzYW1lXG4gICAqIGNvbnRhaW5lci4gRW1iZWRkZWQgdmlld3MgYWxyZWFkeSBoYXZlIGEgbm9kZS5uZXh0LCBidXQgaXQgaXMgb25seSBzZXQgZm9yXG4gICAqIHZpZXdzIGluIHRoZSBzYW1lIGNvbnRhaW5lci4gV2UgbmVlZCBhIHdheSB0byBsaW5rIGNvbXBvbmVudCB2aWV3cyBhbmQgdmlld3NcbiAgICogYWNyb3NzIGNvbnRhaW5lcnMgYXMgd2VsbC5cbiAgICovXG4gIFtORVhUXTogTFZpZXd8TENvbnRhaW5lcnxudWxsO1xuXG4gIC8qKiBRdWVyaWVzIGFjdGl2ZSBmb3IgdGhpcyB2aWV3IC0gbm9kZXMgZnJvbSBhIHZpZXcgYXJlIHJlcG9ydGVkIHRvIHRob3NlIHF1ZXJpZXMuICovXG4gIFtRVUVSSUVTXTogTFF1ZXJpZXN8bnVsbDtcblxuICAvKipcbiAgICogU3RvcmUgdGhlIGBUTm9kZWAgb2YgdGhlIGxvY2F0aW9uIHdoZXJlIHRoZSBjdXJyZW50IGBMVmlld2AgaXMgaW5zZXJ0ZWQgaW50by5cbiAgICpcbiAgICogR2l2ZW46XG4gICAqIGBgYFxuICAgKiA8ZGl2PlxuICAgKiAgIDxuZy10ZW1wbGF0ZT48c3Bhbj48L3NwYW4+PC9uZy10ZW1wbGF0ZT5cbiAgICogPC9kaXY+XG4gICAqIGBgYFxuICAgKlxuICAgKiBXZSBlbmQgdXAgd2l0aCB0d28gYFRWaWV3YHMuXG4gICAqIC0gYHBhcmVudGAgYFRWaWV3YCB3aGljaCBjb250YWlucyBgPGRpdj48IS0tIGFuY2hvciAtLT48L2Rpdj5gXG4gICAqIC0gYGNoaWxkYCBgVFZpZXdgIHdoaWNoIGNvbnRhaW5zIGA8c3Bhbj48L3NwYW4+YFxuICAgKlxuICAgKiBUeXBpY2FsbHkgdGhlIGBjaGlsZGAgaXMgaW5zZXJ0ZWQgaW50byB0aGUgZGVjbGFyYXRpb24gbG9jYXRpb24gb2YgdGhlIGBwYXJlbnRgLCBidXQgaXQgY2FuIGJlXG4gICAqIGluc2VydGVkIGFueXdoZXJlLiBCZWNhdXNlIGl0IGNhbiBiZSBpbnNlcnRlZCBhbnl3aGVyZSBpdCBpcyBub3QgcG9zc2libGUgdG8gc3RvcmUgdGhlXG4gICAqIGluc2VydGlvbiBpbmZvcm1hdGlvbiBpbiB0aGUgYFRWaWV3YCBhbmQgaW5zdGVhZCB3ZSBtdXN0IHN0b3JlIGl0IGluIHRoZSBgTFZpZXdbVF9IT1NUXWAuXG4gICAqXG4gICAqIFNvIHRvIGRldGVybWluZSB3aGVyZSBpcyBvdXIgaW5zZXJ0aW9uIHBhcmVudCB3ZSB3b3VsZCBleGVjdXRlOlxuICAgKiBgYGBcbiAgICogY29uc3QgcGFyZW50TFZpZXcgPSBsVmlld1tQQVJFTlRdO1xuICAgKiBjb25zdCBwYXJlbnRUTm9kZSA9IGxWaWV3W1RfSE9TVF07XG4gICAqIGNvbnN0IGluc2VydGlvblBhcmVudCA9IHBhcmVudExWaWV3W3BhcmVudFROb2RlLmluZGV4XTtcbiAgICogYGBgXG4gICAqXG4gICAqXG4gICAqIElmIGBudWxsYCwgdGhpcyBpcyB0aGUgcm9vdCB2aWV3IG9mIGFuIGFwcGxpY2F0aW9uIChyb290IGNvbXBvbmVudCBpcyBpbiB0aGlzIHZpZXcpIGFuZCBpdCBoYXNcbiAgICogbm8gcGFyZW50cy5cbiAgICovXG4gIFtUX0hPU1RdOiBUTm9kZXxudWxsO1xuXG4gIC8qKlxuICAgKiBXaGVuIGEgdmlldyBpcyBkZXN0cm95ZWQsIGxpc3RlbmVycyBuZWVkIHRvIGJlIHJlbGVhc2VkIGFuZCBvdXRwdXRzIG5lZWQgdG8gYmVcbiAgICogdW5zdWJzY3JpYmVkLiBUaGlzIGNvbnRleHQgYXJyYXkgc3RvcmVzIGJvdGggbGlzdGVuZXIgZnVuY3Rpb25zIHdyYXBwZWQgd2l0aFxuICAgKiB0aGVpciBjb250ZXh0IGFuZCBvdXRwdXQgc3Vic2NyaXB0aW9uIGluc3RhbmNlcyBmb3IgYSBwYXJ0aWN1bGFyIHZpZXcuXG4gICAqXG4gICAqIFRoZXNlIGNoYW5nZSBwZXIgTFZpZXcgaW5zdGFuY2UsIHNvIHRoZXkgY2Fubm90IGJlIHN0b3JlZCBvbiBUVmlldy4gSW5zdGVhZCxcbiAgICogVFZpZXcuY2xlYW51cCBzYXZlcyBhbiBpbmRleCB0byB0aGUgbmVjZXNzYXJ5IGNvbnRleHQgaW4gdGhpcyBhcnJheS5cbiAgICpcbiAgICogQWZ0ZXIgYExWaWV3YCBpcyBjcmVhdGVkIGl0IGlzIHBvc3NpYmxlIHRvIGF0dGFjaCBhZGRpdGlvbmFsIGluc3RhbmNlIHNwZWNpZmljIGZ1bmN0aW9ucyBhdCB0aGVcbiAgICogZW5kIG9mIHRoZSBgbFZpZXdbQ0xFTlVQXWAgYmVjYXVzZSB3ZSBrbm93IHRoYXQgbm8gbW9yZSBgVGAgbGV2ZWwgY2xlYW51cCBmdW5jdGlvbnMgd2lsbCBiZVxuICAgKiBhZGRlZGVkIGhlcmUuXG4gICAqL1xuICBbQ0xFQU5VUF06IGFueVtdfG51bGw7XG5cbiAgLyoqXG4gICAqIC0gRm9yIGR5bmFtaWMgdmlld3MsIHRoaXMgaXMgdGhlIGNvbnRleHQgd2l0aCB3aGljaCB0byByZW5kZXIgdGhlIHRlbXBsYXRlIChlLmcuXG4gICAqICAgYE5nRm9yQ29udGV4dGApLCBvciBge31gIGlmIG5vdCBkZWZpbmVkIGV4cGxpY2l0bHkuXG4gICAqIC0gRm9yIHJvb3QgdmlldyBvZiB0aGUgcm9vdCBjb21wb25lbnQgdGhlIGNvbnRleHQgY29udGFpbnMgY2hhbmdlIGRldGVjdGlvbiBkYXRhLlxuICAgKiAtIEZvciBub24tcm9vdCBjb21wb25lbnRzLCB0aGUgY29udGV4dCBpcyB0aGUgY29tcG9uZW50IGluc3RhbmNlLFxuICAgKiAtIEZvciBpbmxpbmUgdmlld3MsIHRoZSBjb250ZXh0IGlzIG51bGwuXG4gICAqL1xuICBbQ09OVEVYVF06IHt9fFJvb3RDb250ZXh0fG51bGw7XG5cbiAgLyoqIEFuIG9wdGlvbmFsIE1vZHVsZSBJbmplY3RvciB0byBiZSB1c2VkIGFzIGZhbGwgYmFjayBhZnRlciBFbGVtZW50IEluamVjdG9ycyBhcmUgY29uc3VsdGVkLiAqL1xuICByZWFkb25seVtJTkpFQ1RPUl06IEluamVjdG9yfG51bGw7XG5cbiAgLyoqIEZhY3RvcnkgdG8gYmUgdXNlZCBmb3IgY3JlYXRpbmcgUmVuZGVyZXIuICovXG4gIFtSRU5ERVJFUl9GQUNUT1JZXTogUmVuZGVyZXJGYWN0b3J5MztcblxuICAvKiogUmVuZGVyZXIgdG8gYmUgdXNlZCBmb3IgdGhpcyB2aWV3LiAqL1xuICBbUkVOREVSRVJdOiBSZW5kZXJlcjM7XG5cbiAgLyoqIEFuIG9wdGlvbmFsIGN1c3RvbSBzYW5pdGl6ZXIuICovXG4gIFtTQU5JVElaRVJdOiBTYW5pdGl6ZXJ8bnVsbDtcblxuICAvKipcbiAgICogUmVmZXJlbmNlIHRvIHRoZSBmaXJzdCBMVmlldyBvciBMQ29udGFpbmVyIGJlbmVhdGggdGhpcyBMVmlldyBpblxuICAgKiB0aGUgaGllcmFyY2h5LlxuICAgKlxuICAgKiBOZWNlc3NhcnkgdG8gc3RvcmUgdGhpcyBzbyB2aWV3cyBjYW4gdHJhdmVyc2UgdGhyb3VnaCB0aGVpciBuZXN0ZWQgdmlld3NcbiAgICogdG8gcmVtb3ZlIGxpc3RlbmVycyBhbmQgY2FsbCBvbkRlc3Ryb3kgY2FsbGJhY2tzLlxuICAgKi9cbiAgW0NISUxEX0hFQURdOiBMVmlld3xMQ29udGFpbmVyfG51bGw7XG5cbiAgLyoqXG4gICAqIFRoZSBsYXN0IExWaWV3IG9yIExDb250YWluZXIgYmVuZWF0aCB0aGlzIExWaWV3IGluIHRoZSBoaWVyYXJjaHkuXG4gICAqXG4gICAqIFRoZSB0YWlsIGFsbG93cyB1cyB0byBxdWlja2x5IGFkZCBhIG5ldyBzdGF0ZSB0byB0aGUgZW5kIG9mIHRoZSB2aWV3IGxpc3RcbiAgICogd2l0aG91dCBoYXZpbmcgdG8gcHJvcGFnYXRlIHN0YXJ0aW5nIGZyb20gdGhlIGZpcnN0IGNoaWxkLlxuICAgKi9cbiAgW0NISUxEX1RBSUxdOiBMVmlld3xMQ29udGFpbmVyfG51bGw7XG5cbiAgLyoqXG4gICAqIFZpZXcgd2hlcmUgdGhpcyB2aWV3J3MgdGVtcGxhdGUgd2FzIGRlY2xhcmVkLlxuICAgKlxuICAgKiBUaGUgdGVtcGxhdGUgZm9yIGEgZHluYW1pY2FsbHkgY3JlYXRlZCB2aWV3IG1heSBiZSBkZWNsYXJlZCBpbiBhIGRpZmZlcmVudCB2aWV3IHRoYW5cbiAgICogaXQgaXMgaW5zZXJ0ZWQuIFdlIGFscmVhZHkgdHJhY2sgdGhlIFwiaW5zZXJ0aW9uIHZpZXdcIiAodmlldyB3aGVyZSB0aGUgdGVtcGxhdGUgd2FzXG4gICAqIGluc2VydGVkKSBpbiBMVmlld1tQQVJFTlRdLCBidXQgd2UgYWxzbyBuZWVkIGFjY2VzcyB0byB0aGUgXCJkZWNsYXJhdGlvbiB2aWV3XCJcbiAgICogKHZpZXcgd2hlcmUgdGhlIHRlbXBsYXRlIHdhcyBkZWNsYXJlZCkuIE90aGVyd2lzZSwgd2Ugd291bGRuJ3QgYmUgYWJsZSB0byBjYWxsIHRoZVxuICAgKiB2aWV3J3MgdGVtcGxhdGUgZnVuY3Rpb24gd2l0aCB0aGUgcHJvcGVyIGNvbnRleHRzLiBDb250ZXh0IHNob3VsZCBiZSBpbmhlcml0ZWQgZnJvbVxuICAgKiB0aGUgZGVjbGFyYXRpb24gdmlldyB0cmVlLCBub3QgdGhlIGluc2VydGlvbiB2aWV3IHRyZWUuXG4gICAqXG4gICAqIEV4YW1wbGUgKEFwcENvbXBvbmVudCB0ZW1wbGF0ZSk6XG4gICAqXG4gICAqIDxuZy10ZW1wbGF0ZSAjZm9vPjwvbmctdGVtcGxhdGU+ICAgICAgIDwtLSBkZWNsYXJlZCBoZXJlIC0tPlxuICAgKiA8c29tZS1jb21wIFt0cGxdPVwiZm9vXCI+PC9zb21lLWNvbXA+ICAgIDwtLSBpbnNlcnRlZCBpbnNpZGUgdGhpcyBjb21wb25lbnQgLS0+XG4gICAqXG4gICAqIFRoZSA8bmctdGVtcGxhdGU+IGFib3ZlIGlzIGRlY2xhcmVkIGluIHRoZSBBcHBDb21wb25lbnQgdGVtcGxhdGUsIGJ1dCBpdCB3aWxsIGJlIHBhc3NlZCBpbnRvXG4gICAqIFNvbWVDb21wIGFuZCBpbnNlcnRlZCB0aGVyZS4gSW4gdGhpcyBjYXNlLCB0aGUgZGVjbGFyYXRpb24gdmlldyB3b3VsZCBiZSB0aGUgQXBwQ29tcG9uZW50LFxuICAgKiBidXQgdGhlIGluc2VydGlvbiB2aWV3IHdvdWxkIGJlIFNvbWVDb21wLiBXaGVuIHdlIGFyZSByZW1vdmluZyB2aWV3cywgd2Ugd291bGQgd2FudCB0b1xuICAgKiB0cmF2ZXJzZSB0aHJvdWdoIHRoZSBpbnNlcnRpb24gdmlldyB0byBjbGVhbiB1cCBsaXN0ZW5lcnMuIFdoZW4gd2UgYXJlIGNhbGxpbmcgdGhlXG4gICAqIHRlbXBsYXRlIGZ1bmN0aW9uIGR1cmluZyBjaGFuZ2UgZGV0ZWN0aW9uLCB3ZSBuZWVkIHRoZSBkZWNsYXJhdGlvbiB2aWV3IHRvIGdldCBpbmhlcml0ZWRcbiAgICogY29udGV4dC5cbiAgICovXG4gIFtERUNMQVJBVElPTl9WSUVXXTogTFZpZXd8bnVsbDtcblxuXG4gIC8qKlxuICAgKiBQb2ludHMgdG8gdGhlIGRlY2xhcmF0aW9uIGNvbXBvbmVudCB2aWV3LCB1c2VkIHRvIHRyYWNrIHRyYW5zcGxhbnRlZCBgTFZpZXdgcy5cbiAgICpcbiAgICogU2VlOiBgREVDTEFSQVRJT05fVklFV2Agd2hpY2ggcG9pbnRzIHRvIHRoZSBhY3R1YWwgYExWaWV3YCB3aGVyZSBpdCB3YXMgZGVjbGFyZWQsIHdoZXJlYXNcbiAgICogYERFQ0xBUkFUSU9OX0NPTVBPTkVOVF9WSUVXYCBwb2ludHMgdG8gdGhlIGNvbXBvbmVudCB3aGljaCBtYXkgbm90IGJlIHNhbWUgYXNcbiAgICogYERFQ0xBUkFUSU9OX1ZJRVdgLlxuICAgKlxuICAgKiBFeGFtcGxlOlxuICAgKiBgYGBcbiAgICogPCNWSUVXICNteUNvbXA+XG4gICAqICA8ZGl2ICpuZ0lmPVwidHJ1ZVwiPlxuICAgKiAgIDxuZy10ZW1wbGF0ZSAjbXlUbXBsPi4uLjwvbmctdGVtcGxhdGU+XG4gICAqICA8L2Rpdj5cbiAgICogPC8jVklFVz5cbiAgICogYGBgXG4gICAqIEluIHRoZSBhYm92ZSBjYXNlIGBERUNMQVJBVElPTl9WSUVXYCBmb3IgYG15VG1wbGAgcG9pbnRzIHRvIHRoZSBgTFZpZXdgIG9mIGBuZ0lmYCB3aGVyZWFzXG4gICAqIGBERUNMQVJBVElPTl9DT01QT05FTlRfVklFV2AgcG9pbnRzIHRvIGBMVmlld2Agb2YgdGhlIGBteUNvbXBgIHdoaWNoIG93bnMgdGhlIHRlbXBsYXRlLlxuICAgKlxuICAgKiBUaGUgcmVhc29uIGZvciB0aGlzIGlzIHRoYXQgYWxsIGVtYmVkZGVkIHZpZXdzIGFyZSBhbHdheXMgY2hlY2stYWx3YXlzIHdoZXJlYXMgdGhlIGNvbXBvbmVudFxuICAgKiB2aWV3IGNhbiBiZSBjaGVjay1hbHdheXMgb3Igb24tcHVzaC4gV2hlbiB3ZSBoYXZlIGEgdHJhbnNwbGFudGVkIHZpZXcgaXQgaXMgaW1wb3J0YW50IHRvXG4gICAqIGRldGVybWluZSBpZiB3ZSBoYXZlIHRyYW5zcGxhbnRlZCBhIHZpZXcgZnJvbSBjaGVjay1hbHdheXMgZGVjbGFyYXRpb24gdG8gb24tcHVzaCBpbnNlcnRpb25cbiAgICogcG9pbnQuIEluIHN1Y2ggYSBjYXNlIHRoZSB0cmFuc3BsYW50ZWQgdmlldyBuZWVkcyB0byBiZSBhZGRlZCB0byB0aGUgYExDb250YWluZXJgIGluIHRoZVxuICAgKiBkZWNsYXJlZCBgTFZpZXdgIGFuZCBDRCBkdXJpbmcgdGhlIGRlY2xhcmVkIHZpZXcgQ0QgKGluIGFkZGl0aW9uIHRvIHRoZSBDRCBhdCB0aGUgaW5zZXJ0aW9uXG4gICAqIHBvaW50LikgKEFueSB0cmFuc3BsYW50ZWQgdmlld3Mgd2hpY2ggYXJlIGludHJhIENvbXBvbmVudCBhcmUgb2Ygbm8gaW50ZXJlc3QgYmVjYXVzZSB0aGUgQ0RcbiAgICogc3RyYXRlZ3kgb2YgZGVjbGFyYXRpb24gYW5kIGluc2VydGlvbiB3aWxsIGFsd2F5cyBiZSB0aGUgc2FtZSwgYmVjYXVzZSBpdCBpcyB0aGUgc2FtZVxuICAgKiBjb21wb25lbnQuKVxuICAgKlxuICAgKiBRdWVyaWVzIGFscmVhZHkgdHJhY2sgbW92ZWQgdmlld3MgaW4gYExWaWV3W0RFQ0xBUkFUSU9OX0xDT05UQUlORVJdYCBhbmRcbiAgICogYExDb250YWluZXJbTU9WRURfVklFV1NdYC4gSG93ZXZlciB0aGUgcXVlcmllcyBhbHNvIHRyYWNrIGBMVmlld2BzIHdoaWNoIG1vdmVkIHdpdGhpbiB0aGUgc2FtZVxuICAgKiBjb21wb25lbnQgYExWaWV3YC4gVHJhbnNwbGFudGVkIHZpZXdzIGFyZSBhIHN1YnNldCBvZiBtb3ZlZCB2aWV3cywgYW5kIHdlIHVzZVxuICAgKiBgREVDTEFSQVRJT05fQ09NUE9ORU5UX1ZJRVdgIHRvIGRpZmZlcmVudGlhdGUgdGhlbS4gQXMgaW4gdGhpcyBleGFtcGxlLlxuICAgKlxuICAgKiBFeGFtcGxlIHNob3dpbmcgaW50cmEgY29tcG9uZW50IGBMVmlld2AgbW92ZW1lbnQuXG4gICAqIGBgYFxuICAgKiA8I1ZJRVcgI215Q29tcD5cbiAgICogICA8ZGl2ICpuZ0lmPVwiY29uZGl0aW9uOyB0aGVuIHRoZW5CbG9jayBlbHNlIGVsc2VCbG9ja1wiPjwvZGl2PlxuICAgKiAgIDxuZy10ZW1wbGF0ZSAjdGhlbkJsb2NrPkNvbnRlbnQgdG8gcmVuZGVyIHdoZW4gY29uZGl0aW9uIGlzIHRydWUuPC9uZy10ZW1wbGF0ZT5cbiAgICogICA8bmctdGVtcGxhdGUgI2Vsc2VCbG9jaz5Db250ZW50IHRvIHJlbmRlciB3aGVuIGNvbmRpdGlvbiBpcyBmYWxzZS48L25nLXRlbXBsYXRlPlxuICAgKiA8LyNWSUVXPlxuICAgKiBgYGBcbiAgICogVGhlIGB0aGVuQmxvY2tgIGFuZCBgZWxzZUJsb2NrYCBpcyBtb3ZlZCBidXQgbm90IHRyYW5zcGxhbnRlZC5cbiAgICpcbiAgICogRXhhbXBsZSBzaG93aW5nIGludGVyIGNvbXBvbmVudCBgTFZpZXdgIG1vdmVtZW50ICh0cmFuc3BsYW50ZWQgdmlldykuXG4gICAqIGBgYFxuICAgKiA8I1ZJRVcgI215Q29tcD5cbiAgICogICA8bmctdGVtcGxhdGUgI215VG1wbD4uLi48L25nLXRlbXBsYXRlPlxuICAgKiAgIDxpbnNlcnRpb24tY29tcG9uZW50IFt0ZW1wbGF0ZV09XCJteVRtcGxcIj48L2luc2VydGlvbi1jb21wb25lbnQ+XG4gICAqIDwvI1ZJRVc+XG4gICAqIGBgYFxuICAgKiBJbiB0aGUgYWJvdmUgZXhhbXBsZSBgbXlUbXBsYCBpcyBwYXNzZWQgaW50byBhIGRpZmZlcmVudCBjb21wb25lbnQuIElmIGBpbnNlcnRpb24tY29tcG9uZW50YFxuICAgKiBpbnN0YW50aWF0ZXMgYG15VG1wbGAgYW5kIGBpbnNlcnRpb24tY29tcG9uZW50YCBpcyBvbi1wdXNoIHRoZW4gdGhlIGBMQ29udGFpbmVyYCBuZWVkcyB0byBiZVxuICAgKiBtYXJrZWQgYXMgY29udGFpbmluZyB0cmFuc3BsYW50ZWQgdmlld3MgYW5kIHRob3NlIHZpZXdzIG5lZWQgdG8gYmUgQ0QgYXMgcGFydCBvZiB0aGVcbiAgICogZGVjbGFyYXRpb24gQ0QuXG4gICAqXG4gICAqXG4gICAqIFdoZW4gY2hhbmdlIGRldGVjdGlvbiBydW5zLCBpdCBpdGVyYXRlcyBvdmVyIGBbTU9WRURfVklFV1NdYCBhbmQgQ0RzIGFueSBjaGlsZCBgTFZpZXdgcyB3aGVyZVxuICAgKiB0aGUgYERFQ0xBUkFUSU9OX0NPTVBPTkVOVF9WSUVXYCBvZiB0aGUgY3VycmVudCBjb21wb25lbnQgYW5kIHRoZSBjaGlsZCBgTFZpZXdgIGRvZXMgbm90IG1hdGNoXG4gICAqIChpdCBoYXMgYmVlbiB0cmFuc3BsYW50ZWQgYWNyb3NzIGNvbXBvbmVudHMuKVxuICAgKlxuICAgKiBOb3RlOiBgW0RFQ0xBUkFUSU9OX0NPTVBPTkVOVF9WSUVXXWAgcG9pbnRzIHRvIGl0c2VsZiBpZiB0aGUgTFZpZXcgaXMgYSBjb21wb25lbnQgdmlldyAodGhlXG4gICAqICAgICAgIHNpbXBsZXN0IC8gbW9zdCBjb21tb24gY2FzZSkuXG4gICAqXG4gICAqIHNlZSBhbHNvOlxuICAgKiAgIC0gaHR0cHM6Ly9oYWNrbWQuaW8vQG1oZXZlcnkvckpVSnN2djlIIHdyaXRlIHVwIG9mIHRoZSBwcm9ibGVtXG4gICAqICAgLSBgTENvbnRhaW5lcltIQVNfVFJBTlNQTEFOVEVEX1ZJRVdTXWAgd2hpY2ggbWFya3Mgd2hpY2ggYExDb250YWluZXJgIGhhcyB0cmFuc3BsYW50ZWQgdmlld3MuXG4gICAqICAgLSBgTENvbnRhaW5lcltUUkFOU1BMQU5UX0hFQURdYCBhbmQgYExDb250YWluZXJbVFJBTlNQTEFOVF9UQUlMXWAgc3RvcmFnZSBmb3IgdHJhbnNwbGFudGVkXG4gICAqICAgLSBgTFZpZXdbREVDTEFSQVRJT05fTENPTlRBSU5FUl1gIHNpbWlsYXIgcHJvYmxlbSBmb3IgcXVlcmllc1xuICAgKiAgIC0gYExDb250YWluZXJbTU9WRURfVklFV1NdYCBzaW1pbGFyIHByb2JsZW0gZm9yIHF1ZXJpZXNcbiAgICovXG4gIFtERUNMQVJBVElPTl9DT01QT05FTlRfVklFV106IExWaWV3O1xuXG4gIC8qKlxuICAgKiBBIGRlY2xhcmF0aW9uIHBvaW50IG9mIGVtYmVkZGVkIHZpZXdzIChvbmVzIGluc3RhbnRpYXRlZCBiYXNlZCBvbiB0aGUgY29udGVudCBvZiBhXG4gICAqIDxuZy10ZW1wbGF0ZT4pLCBudWxsIGZvciBvdGhlciB0eXBlcyBvZiB2aWV3cy5cbiAgICpcbiAgICogV2UgbmVlZCB0byB0cmFjayBhbGwgZW1iZWRkZWQgdmlld3MgY3JlYXRlZCBmcm9tIGEgZ2l2ZW4gZGVjbGFyYXRpb24gcG9pbnQgc28gd2UgY2FuIHByZXBhcmVcbiAgICogcXVlcnkgbWF0Y2hlcyBpbiBhIHByb3BlciBvcmRlciAocXVlcnkgbWF0Y2hlcyBhcmUgb3JkZXJlZCBiYXNlZCBvbiB0aGVpciBkZWNsYXJhdGlvbiBwb2ludCBhbmRcbiAgICogX25vdF8gdGhlIGluc2VydGlvbiBwb2ludCkuXG4gICAqL1xuICBbREVDTEFSQVRJT05fTENPTlRBSU5FUl06IExDb250YWluZXJ8bnVsbDtcblxuICAvKipcbiAgICogTW9yZSBmbGFncyBmb3IgdGhpcyB2aWV3LiBTZWUgUHJlT3JkZXJIb29rRmxhZ3MgZm9yIG1vcmUgaW5mby5cbiAgICovXG4gIFtQUkVPUkRFUl9IT09LX0ZMQUdTXTogUHJlT3JkZXJIb29rRmxhZ3M7XG5cbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2YgZGlyZWN0IHRyYW5zcGxhbnRlZCB2aWV3cyB3aGljaCBuZWVkIGEgcmVmcmVzaCBvciBoYXZlIGRlc2NlbmRhbnRzIHRoZW1zZWx2ZXNcbiAgICogdGhhdCBuZWVkIGEgcmVmcmVzaCBidXQgaGF2ZSBub3QgbWFya2VkIHRoZWlyIGFuY2VzdG9ycyBhcyBEaXJ0eS4gVGhpcyB0ZWxscyB1cyB0aGF0IGR1cmluZ1xuICAgKiBjaGFuZ2UgZGV0ZWN0aW9uIHdlIHNob3VsZCBzdGlsbCBkZXNjZW5kIHRvIGZpbmQgdGhvc2UgY2hpbGRyZW4gdG8gcmVmcmVzaCwgZXZlbiBpZiB0aGUgcGFyZW50c1xuICAgKiBhcmUgbm90IGBEaXJ0eWAvYENoZWNrQWx3YXlzYC5cbiAgICovXG4gIFtUUkFOU1BMQU5URURfVklFV1NfVE9fUkVGUkVTSF06IG51bWJlcjtcbn1cblxuLyoqIEZsYWdzIGFzc29jaWF0ZWQgd2l0aCBhbiBMVmlldyAoc2F2ZWQgaW4gTFZpZXdbRkxBR1NdKSAqL1xuZXhwb3J0IGNvbnN0IGVudW0gTFZpZXdGbGFncyB7XG4gIC8qKiBUaGUgc3RhdGUgb2YgdGhlIGluaXQgcGhhc2Ugb24gdGhlIGZpcnN0IDIgYml0cyAqL1xuICBJbml0UGhhc2VTdGF0ZUluY3JlbWVudGVyID0gMGIwMDAwMDAwMDAwMSxcbiAgSW5pdFBoYXNlU3RhdGVNYXNrID0gMGIwMDAwMDAwMDAxMSxcblxuICAvKipcbiAgICogV2hldGhlciBvciBub3QgdGhlIHZpZXcgaXMgaW4gY3JlYXRpb25Nb2RlLlxuICAgKlxuICAgKiBUaGlzIG11c3QgYmUgc3RvcmVkIGluIHRoZSB2aWV3IHJhdGhlciB0aGFuIHVzaW5nIGBkYXRhYCBhcyBhIG1hcmtlciBzbyB0aGF0XG4gICAqIHdlIGNhbiBwcm9wZXJseSBzdXBwb3J0IGVtYmVkZGVkIHZpZXdzLiBPdGhlcndpc2UsIHdoZW4gZXhpdGluZyBhIGNoaWxkIHZpZXdcbiAgICogYmFjayBpbnRvIHRoZSBwYXJlbnQgdmlldywgYGRhdGFgIHdpbGwgYmUgZGVmaW5lZCBhbmQgYGNyZWF0aW9uTW9kZWAgd2lsbCBiZVxuICAgKiBpbXByb3Blcmx5IHJlcG9ydGVkIGFzIGZhbHNlLlxuICAgKi9cbiAgQ3JlYXRpb25Nb2RlID0gMGIwMDAwMDAwMDEwMCxcblxuICAvKipcbiAgICogV2hldGhlciBvciBub3QgdGhpcyBMVmlldyBpbnN0YW5jZSBpcyBvbiBpdHMgZmlyc3QgcHJvY2Vzc2luZyBwYXNzLlxuICAgKlxuICAgKiBBbiBMVmlldyBpbnN0YW5jZSBpcyBjb25zaWRlcmVkIHRvIGJlIG9uIGl0cyBcImZpcnN0IHBhc3NcIiB1bnRpbCBpdFxuICAgKiBoYXMgY29tcGxldGVkIG9uZSBjcmVhdGlvbiBtb2RlIHJ1biBhbmQgb25lIHVwZGF0ZSBtb2RlIHJ1bi4gQXQgdGhpc1xuICAgKiB0aW1lLCB0aGUgZmxhZyBpcyB0dXJuZWQgb2ZmLlxuICAgKi9cbiAgRmlyc3RMVmlld1Bhc3MgPSAwYjAwMDAwMDAxMDAwLFxuXG4gIC8qKiBXaGV0aGVyIHRoaXMgdmlldyBoYXMgZGVmYXVsdCBjaGFuZ2UgZGV0ZWN0aW9uIHN0cmF0ZWd5IChjaGVja3MgYWx3YXlzKSBvciBvblB1c2ggKi9cbiAgQ2hlY2tBbHdheXMgPSAwYjAwMDAwMDEwMDAwLFxuXG4gIC8qKlxuICAgKiBXaGV0aGVyIG9yIG5vdCBtYW51YWwgY2hhbmdlIGRldGVjdGlvbiBpcyB0dXJuZWQgb24gZm9yIG9uUHVzaCBjb21wb25lbnRzLlxuICAgKlxuICAgKiBUaGlzIGlzIGEgc3BlY2lhbCBtb2RlIHRoYXQgb25seSBtYXJrcyBjb21wb25lbnRzIGRpcnR5IGluIHR3byBjYXNlczpcbiAgICogMSkgVGhlcmUgaGFzIGJlZW4gYSBjaGFuZ2UgdG8gYW4gQElucHV0IHByb3BlcnR5XG4gICAqIDIpIGBtYXJrRGlydHkoKWAgaGFzIGJlZW4gY2FsbGVkIG1hbnVhbGx5IGJ5IHRoZSB1c2VyXG4gICAqXG4gICAqIE5vdGUgdGhhdCBpbiB0aGlzIG1vZGUsIHRoZSBmaXJpbmcgb2YgZXZlbnRzIGRvZXMgTk9UIG1hcmsgY29tcG9uZW50c1xuICAgKiBkaXJ0eSBhdXRvbWF0aWNhbGx5LlxuICAgKlxuICAgKiBNYW51YWwgbW9kZSBpcyB0dXJuZWQgb2ZmIGJ5IGRlZmF1bHQgZm9yIGJhY2t3YXJkcyBjb21wYXRpYmlsaXR5LCBhcyBldmVudHNcbiAgICogYXV0b21hdGljYWxseSBtYXJrIE9uUHVzaCBjb21wb25lbnRzIGRpcnR5IGluIFZpZXcgRW5naW5lLlxuICAgKlxuICAgKiBUT0RPOiBBZGQgYSBwdWJsaWMgQVBJIHRvIENoYW5nZURldGVjdGlvblN0cmF0ZWd5IHRvIHR1cm4gdGhpcyBtb2RlIG9uXG4gICAqL1xuICBNYW51YWxPblB1c2ggPSAwYjAwMDAwMTAwMDAwLFxuXG4gIC8qKiBXaGV0aGVyIG9yIG5vdCB0aGlzIHZpZXcgaXMgY3VycmVudGx5IGRpcnR5IChuZWVkaW5nIGNoZWNrKSAqL1xuICBEaXJ0eSA9IDBiMDAwMDAxMDAwMDAwLFxuXG4gIC8qKiBXaGV0aGVyIG9yIG5vdCB0aGlzIHZpZXcgaXMgY3VycmVudGx5IGF0dGFjaGVkIHRvIGNoYW5nZSBkZXRlY3Rpb24gdHJlZS4gKi9cbiAgQXR0YWNoZWQgPSAwYjAwMDAxMDAwMDAwMCxcblxuICAvKiogV2hldGhlciBvciBub3QgdGhpcyB2aWV3IGlzIGRlc3Ryb3llZC4gKi9cbiAgRGVzdHJveWVkID0gMGIwMDAxMDAwMDAwMDAsXG5cbiAgLyoqIFdoZXRoZXIgb3Igbm90IHRoaXMgdmlldyBpcyB0aGUgcm9vdCB2aWV3ICovXG4gIElzUm9vdCA9IDBiMDAxMDAwMDAwMDAwLFxuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRoaXMgbW92ZWQgTFZpZXcgd2FzIG5lZWRzIHRvIGJlIHJlZnJlc2hlZCBhdCB0aGUgaW5zZXJ0aW9uIGxvY2F0aW9uIGJlY2F1c2UgdGhlXG4gICAqIGRlY2xhcmF0aW9uIHdhcyBkaXJ0eS5cbiAgICovXG4gIFJlZnJlc2hUcmFuc3BsYW50ZWRWaWV3ID0gMGIwMDEwMDAwMDAwMDAwLFxuXG4gIC8qKlxuICAgKiBJbmRleCBvZiB0aGUgY3VycmVudCBpbml0IHBoYXNlIG9uIGxhc3QgMjEgYml0c1xuICAgKi9cbiAgSW5kZXhXaXRoaW5Jbml0UGhhc2VJbmNyZW1lbnRlciA9IDBiMDEwMDAwMDAwMDAwMCxcbiAgSW5kZXhXaXRoaW5Jbml0UGhhc2VTaGlmdCA9IDExLFxuICBJbmRleFdpdGhpbkluaXRQaGFzZVJlc2V0ID0gMGIwMDExMTExMTExMTExLFxufVxuXG4vKipcbiAqIFBvc3NpYmxlIHN0YXRlcyBvZiB0aGUgaW5pdCBwaGFzZTpcbiAqIC0gMDA6IE9uSW5pdCBob29rcyB0byBiZSBydW4uXG4gKiAtIDAxOiBBZnRlckNvbnRlbnRJbml0IGhvb2tzIHRvIGJlIHJ1blxuICogLSAxMDogQWZ0ZXJWaWV3SW5pdCBob29rcyB0byBiZSBydW5cbiAqIC0gMTE6IEFsbCBpbml0IGhvb2tzIGhhdmUgYmVlbiBydW5cbiAqL1xuZXhwb3J0IGNvbnN0IGVudW0gSW5pdFBoYXNlU3RhdGUge1xuICBPbkluaXRIb29rc1RvQmVSdW4gPSAwYjAwLFxuICBBZnRlckNvbnRlbnRJbml0SG9va3NUb0JlUnVuID0gMGIwMSxcbiAgQWZ0ZXJWaWV3SW5pdEhvb2tzVG9CZVJ1biA9IDBiMTAsXG4gIEluaXRQaGFzZUNvbXBsZXRlZCA9IDBiMTEsXG59XG5cbi8qKiBNb3JlIGZsYWdzIGFzc29jaWF0ZWQgd2l0aCBhbiBMVmlldyAoc2F2ZWQgaW4gTFZpZXdbUFJFT1JERVJfSE9PS19GTEFHU10pICovXG5leHBvcnQgY29uc3QgZW51bSBQcmVPcmRlckhvb2tGbGFncyB7XG4gIC8qKlxuICAgICBUaGUgaW5kZXggb2YgdGhlIG5leHQgcHJlLW9yZGVyIGhvb2sgdG8gYmUgY2FsbGVkIGluIHRoZSBob29rcyBhcnJheSwgb24gdGhlIGZpcnN0IDE2XG4gICAgIGJpdHNcbiAgICovXG4gIEluZGV4T2ZUaGVOZXh0UHJlT3JkZXJIb29rTWFza01hc2sgPSAwYjAxMTExMTExMTExMTExMTExLFxuXG4gIC8qKlxuICAgKiBUaGUgbnVtYmVyIG9mIGluaXQgaG9va3MgdGhhdCBoYXZlIGFscmVhZHkgYmVlbiBjYWxsZWQsIG9uIHRoZSBsYXN0IDE2IGJpdHNcbiAgICovXG4gIE51bWJlck9mSW5pdEhvb2tzQ2FsbGVkSW5jcmVtZW50ZXIgPSAwYjAxMDAwMDAwMDAwMDAwMDAwMCxcbiAgTnVtYmVyT2ZJbml0SG9va3NDYWxsZWRTaGlmdCA9IDE2LFxuICBOdW1iZXJPZkluaXRIb29rc0NhbGxlZE1hc2sgPSAwYjExMTExMTExMTExMTExMTEwMDAwMDAwMDAwMDAwMDAwLFxufVxuXG4vKipcbiAqIFN0b3JlcyBhIHNldCBvZiBPcENvZGVzIHRvIHByb2Nlc3MgYEhvc3RCaW5kaW5nc0Z1bmN0aW9uYCBhc3NvY2lhdGVkIHdpdGggYSBjdXJyZW50IHZpZXcuXG4gKlxuICogSW4gb3JkZXIgdG8gaW52b2tlIGBIb3N0QmluZGluZ3NGdW5jdGlvbmAgd2UgbmVlZDpcbiAqIDEuICdlbGVtZW50SWR4YDogSW5kZXggdG8gdGhlIGVsZW1lbnQgYXNzb2NpYXRlZCB3aXRoIHRoZSBgSG9zdEJpbmRpbmdzRnVuY3Rpb25gLlxuICogMi4gJ2RpcmVjdGl2ZUlkeGA6IEluZGV4IHRvIHRoZSBkaXJlY3RpdmUgYXNzb2NpYXRlZCB3aXRoIHRoZSBgSG9zdEJpbmRpbmdzRnVuY3Rpb25gLiAoVGhpcyB3aWxsXG4gKiAgICBiZWNvbWUgdGhlIGNvbnRleHQgZm9yIHRoZSBgSG9zdEJpbmRpbmdzRnVuY3Rpb25gIGludm9jYXRpb24uKVxuICogMy4gYGJpbmRpbmdSb290SWR4YDogTG9jYXRpb24gd2hlcmUgdGhlIGJpbmRpbmdzIGZvciB0aGUgYEhvc3RCaW5kaW5nc0Z1bmN0aW9uYCBzdGFydC4gSW50ZXJuYWxseVxuICogICAgYEhvc3RCaW5kaW5nc0Z1bmN0aW9uYCBiaW5kaW5nIGluZGV4ZXMgc3RhcnQgZnJvbSBgMGAgc28gd2UgbmVlZCB0byBhZGQgYGJpbmRpbmdSb290SWR4YCB0b1xuICogICAgaXQuXG4gKiA0LiBgSG9zdEJpbmRpbmdzRnVuY3Rpb25gOiBBIGhvc3QgYmluZGluZyBmdW5jdGlvbiB0byBleGVjdXRlLlxuICpcbiAqIFRoZSBhYm92ZSBpbmZvcm1hdGlvbiBuZWVkcyB0byBiZSBlbmNvZGVkIGludG8gdGhlIGBIb3N0QmluZGluZ09wQ29kZXNgIGluIGFuIGVmZmljaWVudCBtYW5uZXIuXG4gKlxuICogMS4gYGVsZW1lbnRJZHhgIGlzIGVuY29kZWQgaW50byB0aGUgYEhvc3RCaW5kaW5nT3BDb2Rlc2AgYXMgYH5lbGVtZW50SWR4YCAoc28gYSBuZWdhdGl2ZSBudW1iZXIpO1xuICogMi4gYGRpcmVjdGl2ZUlkeGBcbiAqIDMuIGBiaW5kaW5nUm9vdElkeGBcbiAqIDQuIGBIb3N0QmluZGluZ3NGdW5jdGlvbmAgaXMgcGFzc2VkIGluIGFzIGlzLlxuICpcbiAqIFRoZSBgSG9zdEJpbmRpbmdPcENvZGVzYCBhcnJheSBjb250YWluczpcbiAqIC0gbmVnYXRpdmUgbnVtYmVyIHRvIHNlbGVjdCB0aGUgZWxlbWVudCBpbmRleC5cbiAqIC0gZm9sbG93ZWQgYnkgMSBvciBtb3JlIG9mOlxuICogICAgLSBhIG51bWJlciB0byBzZWxlY3QgdGhlIGRpcmVjdGl2ZSBpbmRleFxuICogICAgLSBhIG51bWJlciB0byBzZWxlY3QgdGhlIGJpbmRpbmdSb290IGluZGV4XG4gKiAgICAtIGFuZCBhIGZ1bmN0aW9uIHRvIGludm9rZS5cbiAqXG4gKiAjIyBFeGFtcGxlXG4gKlxuICogYGBgXG4gKiBjb25zdCBob3N0QmluZGluZ09wQ29kZXMgPSBbXG4gKiAgIH4zMCwgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgLy8gU2VsZWN0IGVsZW1lbnQgMzBcbiAqICAgNDAsIDQ1LCBNeURpci7JtWRpci5ob3N0QmluZGluZ3MgICAgLy8gSW52b2tlIGhvc3QgYmluZGluZ3Mgb24gTXlEaXIgb24gZWxlbWVudCAzMDtcbiAqICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAvLyBkaXJlY3RpdmVJZHggPSA0MDsgYmluZGluZ1Jvb3RJZHggPSA0NTtcbiAqICAgNTAsIDU1LCBPdGhlckRpci7JtWRpci5ob3N0QmluZGluZ3MgLy8gSW52b2tlIGhvc3QgYmluZGluZ3Mgb24gT3RoZXJEaXJlIG9uIGVsZW1lbnQgMzBcbiAqICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAvLyBkaXJlY3RpdmVJZHggPSA1MDsgYmluZGluZ1Jvb3RJZHggPSA1NTtcbiAqIF1cbiAqIGBgYFxuICpcbiAqICMjIFBzZXVkb2NvZGVcbiAqIGBgYFxuICogY29uc3QgaG9zdEJpbmRpbmdPcENvZGVzID0gdFZpZXcuaG9zdEJpbmRpbmdPcENvZGVzO1xuICogaWYgKGhvc3RCaW5kaW5nT3BDb2RlcyA9PT0gbnVsbCkgcmV0dXJuO1xuICogZm9yIChsZXQgaSA9IDA7IGkgPCBob3N0QmluZGluZ09wQ29kZXMubGVuZ3RoOyBpKyspIHtcbiAqICAgY29uc3Qgb3BDb2RlID0gaG9zdEJpbmRpbmdPcENvZGVzW2ldIGFzIG51bWJlcjtcbiAqICAgaWYgKG9wQ29kZSA8IDApIHtcbiAqICAgICAvLyBOZWdhdGl2ZSBudW1iZXJzIGFyZSBlbGVtZW50IGluZGV4ZXMuXG4gKiAgICAgc2V0U2VsZWN0ZWRJbmRleCh+b3BDb2RlKTtcbiAqICAgfSBlbHNlIHtcbiAqICAgICAvLyBQb3NpdGl2ZSBudW1iZXJzIGFyZSBOdW1iZXJUdXBsZSB3aGljaCBzdG9yZSBiaW5kaW5nUm9vdEluZGV4IGFuZCBkaXJlY3RpdmVJbmRleC5cbiAqICAgICBjb25zdCBkaXJlY3RpdmVJZHggPSBvcENvZGU7XG4gKiAgICAgY29uc3QgYmluZGluZ1Jvb3RJbmR4ID0gaG9zdEJpbmRpbmdPcENvZGVzWysraV0gYXMgbnVtYmVyO1xuICogICAgIGNvbnN0IGhvc3RCaW5kaW5nRm4gPSBob3N0QmluZGluZ09wQ29kZXNbKytpXSBhcyBIb3N0QmluZGluZ3NGdW5jdGlvbjxhbnk+O1xuICogICAgIHNldEJpbmRpbmdSb290Rm9ySG9zdEJpbmRpbmdzKGJpbmRpbmdSb290SW5keCwgZGlyZWN0aXZlSWR4KTtcbiAqICAgICBjb25zdCBjb250ZXh0ID0gbFZpZXdbZGlyZWN0aXZlSWR4XTtcbiAqICAgICBob3N0QmluZGluZ0ZuKFJlbmRlckZsYWdzLlVwZGF0ZSwgY29udGV4dCk7XG4gKiAgIH1cbiAqIH1cbiAqIGBgYFxuICpcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBIb3N0QmluZGluZ09wQ29kZXMgZXh0ZW5kcyBBcnJheTxudW1iZXJ8SG9zdEJpbmRpbmdzRnVuY3Rpb248YW55Pj4ge1xuICBfX2JyYW5kX186ICdIb3N0QmluZGluZ09wQ29kZXMnO1xuICBkZWJ1Zz86IHN0cmluZ1tdO1xufVxuXG4vKipcbiAqIEV4cGxpY2l0bHkgbWFya3MgYFRWaWV3YCBhcyBhIHNwZWNpZmljIHR5cGUgaW4gYG5nRGV2TW9kZWBcbiAqXG4gKiBJdCBpcyB1c2VmdWwgdG8ga25vdyBjb25jZXB0dWFsbHkgd2hhdCB0aW1lIG9mIGBUVmlld2Agd2UgYXJlIGRlYWxpbmcgd2l0aCB3aGVuXG4gKiBkZWJ1Z2dpbmcgYW4gYXBwbGljYXRpb24gKGV2ZW4gaWYgdGhlIHJ1bnRpbWUgZG9lcyBub3QgbmVlZCBpdC4pIEZvciB0aGlzIHJlYXNvblxuICogd2Ugc3RvcmUgdGhpcyBpbmZvcm1hdGlvbiBpbiB0aGUgYG5nRGV2TW9kZWAgYFRWaWV3YCBhbmQgdGhhbiB1c2UgaXQgZm9yXG4gKiBiZXR0ZXIgZGVidWdnaW5nIGV4cGVyaWVuY2UuXG4gKi9cbmV4cG9ydCBjb25zdCBlbnVtIFRWaWV3VHlwZSB7XG4gIC8qKlxuICAgKiBSb290IGBUVmlld2AgaXMgdGhlIHVzZWQgdG8gYm9vdHN0cmFwIGNvbXBvbmVudHMgaW50by4gSXQgaXMgdXNlZCBpbiBjb25qdW5jdGlvbiB3aXRoXG4gICAqIGBMVmlld2Agd2hpY2ggdGFrZXMgYW4gZXhpc3RpbmcgRE9NIG5vZGUgbm90IG93bmVkIGJ5IEFuZ3VsYXIgYW5kIHdyYXBzIGl0IGluIGBUVmlld2AvYExWaWV3YFxuICAgKiBzbyB0aGF0IG90aGVyIGNvbXBvbmVudHMgY2FuIGJlIGxvYWRlZCBpbnRvIGl0LlxuICAgKi9cbiAgUm9vdCA9IDAsXG5cbiAgLyoqXG4gICAqIGBUVmlld2AgYXNzb2NpYXRlZCB3aXRoIGEgQ29tcG9uZW50LiBUaGlzIHdvdWxkIGJlIHRoZSBgVFZpZXdgIGRpcmVjdGx5IGFzc29jaWF0ZWQgd2l0aCB0aGVcbiAgICogY29tcG9uZW50IHZpZXcgKGFzIG9wcG9zZWQgYW4gYEVtYmVkZGVkYCBgVFZpZXdgIHdoaWNoIHdvdWxkIGJlIGEgY2hpbGQgb2YgYENvbXBvbmVudGAgYFRWaWV3YClcbiAgICovXG4gIENvbXBvbmVudCA9IDEsXG5cbiAgLyoqXG4gICAqIGBUVmlld2AgYXNzb2NpYXRlZCB3aXRoIGEgdGVtcGxhdGUuIFN1Y2ggYXMgYCpuZ0lmYCwgYDxuZy10ZW1wbGF0ZT5gIGV0Yy4uLiBBIGBDb21wb25lbnRgXG4gICAqIGNhbiBoYXZlIHplcm8gb3IgbW9yZSBgRW1iZWRlZGVgIGBUVmlld2BzLlxuICAgKi9cbiAgRW1iZWRkZWQgPSAyLFxufVxuXG4vKipcbiAqIENvbnZlcnRzIGBUVmlld1R5cGVgIGludG8gaHVtYW4gcmVhZGFibGUgdGV4dC5cbiAqIE1ha2Ugc3VyZSB0aGlzIG1hdGNoZXMgd2l0aCBgVFZpZXdUeXBlYFxuICovXG5leHBvcnQgY29uc3QgVFZpZXdUeXBlQXNTdHJpbmcgPSBbXG4gICdSb290JywgICAgICAgLy8gMFxuICAnQ29tcG9uZW50JywgIC8vIDFcbiAgJ0VtYmVkZGVkJywgICAvLyAyXG5dIGFzIGNvbnN0O1xuXG5cbi8qKlxuICogVGhlIHN0YXRpYyBkYXRhIGZvciBhbiBMVmlldyAoc2hhcmVkIGJldHdlZW4gYWxsIHRlbXBsYXRlcyBvZiBhXG4gKiBnaXZlbiB0eXBlKS5cbiAqXG4gKiBTdG9yZWQgb24gdGhlIGBDb21wb25lbnREZWYudFZpZXdgLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFRWaWV3IHtcbiAgLyoqXG4gICAqIFR5cGUgb2YgYFRWaWV3YCAoYFJvb3RgfGBDb21wb25lbnRgfGBFbWJlZGRlZGApLlxuICAgKi9cbiAgdHlwZTogVFZpZXdUeXBlO1xuXG4gIC8qKlxuICAgKiBUaGlzIGlzIGEgYmx1ZXByaW50IHVzZWQgdG8gZ2VuZXJhdGUgTFZpZXcgaW5zdGFuY2VzIGZvciB0aGlzIFRWaWV3LiBDb3B5aW5nIHRoaXNcbiAgICogYmx1ZXByaW50IGlzIGZhc3RlciB0aGFuIGNyZWF0aW5nIGEgbmV3IExWaWV3IGZyb20gc2NyYXRjaC5cbiAgICovXG4gIGJsdWVwcmludDogTFZpZXc7XG5cbiAgLyoqXG4gICAqIFRoZSB0ZW1wbGF0ZSBmdW5jdGlvbiB1c2VkIHRvIHJlZnJlc2ggdGhlIHZpZXcgb2YgZHluYW1pY2FsbHkgY3JlYXRlZCB2aWV3c1xuICAgKiBhbmQgY29tcG9uZW50cy4gV2lsbCBiZSBudWxsIGZvciBpbmxpbmUgdmlld3MuXG4gICAqL1xuICB0ZW1wbGF0ZTogQ29tcG9uZW50VGVtcGxhdGU8e30+fG51bGw7XG5cbiAgLyoqXG4gICAqIEEgZnVuY3Rpb24gY29udGFpbmluZyBxdWVyeS1yZWxhdGVkIGluc3RydWN0aW9ucy5cbiAgICovXG4gIHZpZXdRdWVyeTogVmlld1F1ZXJpZXNGdW5jdGlvbjx7fT58bnVsbDtcblxuICAvKipcbiAgICogQSBgVE5vZGVgIHJlcHJlc2VudGluZyB0aGUgZGVjbGFyYXRpb24gbG9jYXRpb24gb2YgdGhpcyBgVFZpZXdgIChub3QgcGFydCBvZiB0aGlzIFRWaWV3KS5cbiAgICovXG4gIGRlY2xUTm9kZTogVE5vZGV8bnVsbDtcblxuICAvLyBGSVhNRShtaXNrbyk6IFdoeSBkb2VzIGBUVmlld2Agbm90IGhhdmUgYGRlY2xhcmF0aW9uVFZpZXdgIHByb3BlcnR5P1xuXG4gIC8qKiBXaGV0aGVyIG9yIG5vdCB0aGlzIHRlbXBsYXRlIGhhcyBiZWVuIHByb2Nlc3NlZCBpbiBjcmVhdGlvbiBtb2RlLiAqL1xuICBmaXJzdENyZWF0ZVBhc3M6IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqICBXaGV0aGVyIG9yIG5vdCB0aGlzIHRlbXBsYXRlIGhhcyBiZWVuIHByb2Nlc3NlZCBpbiB1cGRhdGUgbW9kZSAoZS5nLiBjaGFuZ2UgZGV0ZWN0ZWQpXG4gICAqXG4gICAqIGBmaXJzdFVwZGF0ZVBhc3NgIGlzIHVzZWQgYnkgc3R5bGluZyB0byBzZXQgdXAgYFREYXRhYCB0byBjb250YWluIG1ldGFkYXRhIGFib3V0IHRoZSBzdHlsaW5nXG4gICAqIGluc3RydWN0aW9ucy4gKE1haW5seSB0byBidWlsZCB1cCBhIGxpbmtlZCBsaXN0IG9mIHN0eWxpbmcgcHJpb3JpdHkgb3JkZXIuKVxuICAgKlxuICAgKiBUeXBpY2FsbHkgdGhpcyBmdW5jdGlvbiBnZXRzIGNsZWFyZWQgYWZ0ZXIgZmlyc3QgZXhlY3V0aW9uLiBJZiBleGNlcHRpb24gaXMgdGhyb3duIHRoZW4gdGhpc1xuICAgKiBmbGFnIGNhbiByZW1haW4gdHVybmVkIHVuIHVudGlsIHRoZXJlIGlzIGZpcnN0IHN1Y2Nlc3NmdWwgKG5vIGV4Y2VwdGlvbikgcGFzcy4gVGhpcyBtZWFucyB0aGF0XG4gICAqIGluZGl2aWR1YWwgc3R5bGluZyBpbnN0cnVjdGlvbnMga2VlcCB0cmFjayBvZiBpZiB0aGV5IGhhdmUgYWxyZWFkeSBiZWVuIGFkZGVkIHRvIHRoZSBsaW5rZWRcbiAgICogbGlzdCB0byBwcmV2ZW50IGRvdWJsZSBhZGRpbmcuXG4gICAqL1xuICBmaXJzdFVwZGF0ZVBhc3M6IGJvb2xlYW47XG5cbiAgLyoqIFN0YXRpYyBkYXRhIGVxdWl2YWxlbnQgb2YgTFZpZXcuZGF0YVtdLiBDb250YWlucyBUTm9kZXMsIFBpcGVEZWZJbnRlcm5hbCBvciBUSTE4bi4gKi9cbiAgZGF0YTogVERhdGE7XG5cbiAgLyoqXG4gICAqIFRoZSBiaW5kaW5nIHN0YXJ0IGluZGV4IGlzIHRoZSBpbmRleCBhdCB3aGljaCB0aGUgZGF0YSBhcnJheVxuICAgKiBzdGFydHMgdG8gc3RvcmUgYmluZGluZ3Mgb25seS4gU2F2aW5nIHRoaXMgdmFsdWUgZW5zdXJlcyB0aGF0IHdlXG4gICAqIHdpbGwgYmVnaW4gcmVhZGluZyBiaW5kaW5ncyBhdCB0aGUgY29ycmVjdCBwb2ludCBpbiB0aGUgYXJyYXkgd2hlblxuICAgKiB3ZSBhcmUgaW4gdXBkYXRlIG1vZGUuXG4gICAqXG4gICAqIC0xIG1lYW5zIHRoYXQgaXQgaGFzIG5vdCBiZWVuIGluaXRpYWxpemVkLlxuICAgKi9cbiAgYmluZGluZ1N0YXJ0SW5kZXg6IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIGluZGV4IHdoZXJlIHRoZSBcImV4cGFuZG9cIiBzZWN0aW9uIG9mIGBMVmlld2AgYmVnaW5zLiBUaGUgZXhwYW5kb1xuICAgKiBzZWN0aW9uIGNvbnRhaW5zIGluamVjdG9ycywgZGlyZWN0aXZlIGluc3RhbmNlcywgYW5kIGhvc3QgYmluZGluZyB2YWx1ZXMuXG4gICAqIFVubGlrZSB0aGUgXCJkZWNsc1wiIGFuZCBcInZhcnNcIiBzZWN0aW9ucyBvZiBgTFZpZXdgLCB0aGUgbGVuZ3RoIG9mIHRoaXNcbiAgICogc2VjdGlvbiBjYW5ub3QgYmUgY2FsY3VsYXRlZCBhdCBjb21waWxlLXRpbWUgYmVjYXVzZSBkaXJlY3RpdmVzIGFyZSBtYXRjaGVkXG4gICAqIGF0IHJ1bnRpbWUgdG8gcHJlc2VydmUgbG9jYWxpdHkuXG4gICAqXG4gICAqIFdlIHN0b3JlIHRoaXMgc3RhcnQgaW5kZXggc28gd2Uga25vdyB3aGVyZSB0byBzdGFydCBjaGVja2luZyBob3N0IGJpbmRpbmdzXG4gICAqIGluIGBzZXRIb3N0QmluZGluZ3NgLlxuICAgKi9cbiAgZXhwYW5kb1N0YXJ0SW5kZXg6IG51bWJlcjtcblxuICAvKipcbiAgICogV2hldGhlciBvciBub3QgdGhlcmUgYXJlIGFueSBzdGF0aWMgdmlldyBxdWVyaWVzIHRyYWNrZWQgb24gdGhpcyB2aWV3LlxuICAgKlxuICAgKiBXZSBzdG9yZSB0aGlzIHNvIHdlIGtub3cgd2hldGhlciBvciBub3Qgd2Ugc2hvdWxkIGRvIGEgdmlldyBxdWVyeVxuICAgKiByZWZyZXNoIGFmdGVyIGNyZWF0aW9uIG1vZGUgdG8gY29sbGVjdCBzdGF0aWMgcXVlcnkgcmVzdWx0cy5cbiAgICovXG4gIHN0YXRpY1ZpZXdRdWVyaWVzOiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIG9yIG5vdCB0aGVyZSBhcmUgYW55IHN0YXRpYyBjb250ZW50IHF1ZXJpZXMgdHJhY2tlZCBvbiB0aGlzIHZpZXcuXG4gICAqXG4gICAqIFdlIHN0b3JlIHRoaXMgc28gd2Uga25vdyB3aGV0aGVyIG9yIG5vdCB3ZSBzaG91bGQgZG8gYSBjb250ZW50IHF1ZXJ5XG4gICAqIHJlZnJlc2ggYWZ0ZXIgY3JlYXRpb24gbW9kZSB0byBjb2xsZWN0IHN0YXRpYyBxdWVyeSByZXN1bHRzLlxuICAgKi9cbiAgc3RhdGljQ29udGVudFF1ZXJpZXM6IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEEgcmVmZXJlbmNlIHRvIHRoZSBmaXJzdCBjaGlsZCBub2RlIGxvY2F0ZWQgaW4gdGhlIHZpZXcuXG4gICAqL1xuICBmaXJzdENoaWxkOiBUTm9kZXxudWxsO1xuXG4gIC8qKlxuICAgKiBTdG9yZXMgdGhlIE9wQ29kZXMgdG8gYmUgcmVwbGF5ZWQgZHVyaW5nIGNoYW5nZS1kZXRlY3Rpb24gdG8gcHJvY2VzcyB0aGUgYEhvc3RCaW5kaW5nc2BcbiAgICpcbiAgICogU2VlIGBIb3N0QmluZGluZ09wQ29kZXNgIGZvciBlbmNvZGluZyBkZXRhaWxzLlxuICAgKi9cbiAgaG9zdEJpbmRpbmdPcENvZGVzOiBIb3N0QmluZGluZ09wQ29kZXN8bnVsbDtcblxuICAvKipcbiAgICogRnVsbCByZWdpc3RyeSBvZiBkaXJlY3RpdmVzIGFuZCBjb21wb25lbnRzIHRoYXQgbWF5IGJlIGZvdW5kIGluIHRoaXMgdmlldy5cbiAgICpcbiAgICogSXQncyBuZWNlc3NhcnkgdG8ga2VlcCBhIGNvcHkgb2YgdGhlIGZ1bGwgZGVmIGxpc3Qgb24gdGhlIFRWaWV3IHNvIGl0J3MgcG9zc2libGVcbiAgICogdG8gcmVuZGVyIHRlbXBsYXRlIGZ1bmN0aW9ucyB3aXRob3V0IGEgaG9zdCBjb21wb25lbnQuXG4gICAqL1xuICBkaXJlY3RpdmVSZWdpc3RyeTogRGlyZWN0aXZlRGVmTGlzdHxudWxsO1xuXG4gIC8qKlxuICAgKiBGdWxsIHJlZ2lzdHJ5IG9mIHBpcGVzIHRoYXQgbWF5IGJlIGZvdW5kIGluIHRoaXMgdmlldy5cbiAgICpcbiAgICogVGhlIHByb3BlcnR5IGlzIGVpdGhlciBhbiBhcnJheSBvZiBgUGlwZURlZnNgcyBvciBhIGZ1bmN0aW9uIHdoaWNoIHJldHVybnMgdGhlIGFycmF5IG9mXG4gICAqIGBQaXBlRGVmc2BzLiBUaGUgZnVuY3Rpb24gaXMgbmVjZXNzYXJ5IHRvIGJlIGFibGUgdG8gc3VwcG9ydCBmb3J3YXJkIGRlY2xhcmF0aW9ucy5cbiAgICpcbiAgICogSXQncyBuZWNlc3NhcnkgdG8ga2VlcCBhIGNvcHkgb2YgdGhlIGZ1bGwgZGVmIGxpc3Qgb24gdGhlIFRWaWV3IHNvIGl0J3MgcG9zc2libGVcbiAgICogdG8gcmVuZGVyIHRlbXBsYXRlIGZ1bmN0aW9ucyB3aXRob3V0IGEgaG9zdCBjb21wb25lbnQuXG4gICAqL1xuICBwaXBlUmVnaXN0cnk6IFBpcGVEZWZMaXN0fG51bGw7XG5cbiAgLyoqXG4gICAqIEFycmF5IG9mIG5nT25Jbml0LCBuZ09uQ2hhbmdlcyBhbmQgbmdEb0NoZWNrIGhvb2tzIHRoYXQgc2hvdWxkIGJlIGV4ZWN1dGVkIGZvciB0aGlzIHZpZXcgaW5cbiAgICogY3JlYXRpb24gbW9kZS5cbiAgICpcbiAgICogVGhpcyBhcnJheSBoYXMgYSBmbGF0IHN0cnVjdHVyZSBhbmQgY29udGFpbnMgVE5vZGUgaW5kaWNlcywgZGlyZWN0aXZlIGluZGljZXMgKHdoZXJlIGFuXG4gICAqIGluc3RhbmNlIGNhbiBiZSBmb3VuZCBpbiBgTFZpZXdgKSBhbmQgaG9vayBmdW5jdGlvbnMuIFROb2RlIGluZGV4IGlzIGZvbGxvd2VkIGJ5IHRoZSBkaXJlY3RpdmVcbiAgICogaW5kZXggYW5kIGEgaG9vayBmdW5jdGlvbi4gSWYgdGhlcmUgYXJlIG11bHRpcGxlIGhvb2tzIGZvciBhIGdpdmVuIFROb2RlLCB0aGUgVE5vZGUgaW5kZXggaXNcbiAgICogbm90IHJlcGVhdGVkIGFuZCB0aGUgbmV4dCBsaWZlY3ljbGUgaG9vayBpbmZvcm1hdGlvbiBpcyBzdG9yZWQgcmlnaHQgYWZ0ZXIgdGhlIHByZXZpb3VzIGhvb2tcbiAgICogZnVuY3Rpb24uIFRoaXMgaXMgZG9uZSBzbyB0aGF0IGF0IHJ1bnRpbWUgdGhlIHN5c3RlbSBjYW4gZWZmaWNpZW50bHkgaXRlcmF0ZSBvdmVyIGFsbCBvZiB0aGVcbiAgICogZnVuY3Rpb25zIHRvIGludm9rZSB3aXRob3V0IGhhdmluZyB0byBtYWtlIGFueSBkZWNpc2lvbnMvbG9va3Vwcy5cbiAgICovXG4gIHByZU9yZGVySG9va3M6IEhvb2tEYXRhfG51bGw7XG5cbiAgLyoqXG4gICAqIEFycmF5IG9mIG5nT25DaGFuZ2VzIGFuZCBuZ0RvQ2hlY2sgaG9va3MgdGhhdCBzaG91bGQgYmUgZXhlY3V0ZWQgZm9yIHRoaXMgdmlldyBpbiB1cGRhdGUgbW9kZS5cbiAgICpcbiAgICogVGhpcyBhcnJheSBoYXMgdGhlIHNhbWUgc3RydWN0dXJlIGFzIHRoZSBgcHJlT3JkZXJIb29rc2Agb25lLlxuICAgKi9cbiAgcHJlT3JkZXJDaGVja0hvb2tzOiBIb29rRGF0YXxudWxsO1xuXG4gIC8qKlxuICAgKiBBcnJheSBvZiBuZ0FmdGVyQ29udGVudEluaXQgYW5kIG5nQWZ0ZXJDb250ZW50Q2hlY2tlZCBob29rcyB0aGF0IHNob3VsZCBiZSBleGVjdXRlZFxuICAgKiBmb3IgdGhpcyB2aWV3IGluIGNyZWF0aW9uIG1vZGUuXG4gICAqXG4gICAqIEV2ZW4gaW5kaWNlczogRGlyZWN0aXZlIGluZGV4XG4gICAqIE9kZCBpbmRpY2VzOiBIb29rIGZ1bmN0aW9uXG4gICAqL1xuICBjb250ZW50SG9va3M6IEhvb2tEYXRhfG51bGw7XG5cbiAgLyoqXG4gICAqIEFycmF5IG9mIG5nQWZ0ZXJDb250ZW50Q2hlY2tlZCBob29rcyB0aGF0IHNob3VsZCBiZSBleGVjdXRlZCBmb3IgdGhpcyB2aWV3IGluIHVwZGF0ZVxuICAgKiBtb2RlLlxuICAgKlxuICAgKiBFdmVuIGluZGljZXM6IERpcmVjdGl2ZSBpbmRleFxuICAgKiBPZGQgaW5kaWNlczogSG9vayBmdW5jdGlvblxuICAgKi9cbiAgY29udGVudENoZWNrSG9va3M6IEhvb2tEYXRhfG51bGw7XG5cbiAgLyoqXG4gICAqIEFycmF5IG9mIG5nQWZ0ZXJWaWV3SW5pdCBhbmQgbmdBZnRlclZpZXdDaGVja2VkIGhvb2tzIHRoYXQgc2hvdWxkIGJlIGV4ZWN1dGVkIGZvclxuICAgKiB0aGlzIHZpZXcgaW4gY3JlYXRpb24gbW9kZS5cbiAgICpcbiAgICogRXZlbiBpbmRpY2VzOiBEaXJlY3RpdmUgaW5kZXhcbiAgICogT2RkIGluZGljZXM6IEhvb2sgZnVuY3Rpb25cbiAgICovXG4gIHZpZXdIb29rczogSG9va0RhdGF8bnVsbDtcblxuICAvKipcbiAgICogQXJyYXkgb2YgbmdBZnRlclZpZXdDaGVja2VkIGhvb2tzIHRoYXQgc2hvdWxkIGJlIGV4ZWN1dGVkIGZvciB0aGlzIHZpZXcgaW5cbiAgICogdXBkYXRlIG1vZGUuXG4gICAqXG4gICAqIEV2ZW4gaW5kaWNlczogRGlyZWN0aXZlIGluZGV4XG4gICAqIE9kZCBpbmRpY2VzOiBIb29rIGZ1bmN0aW9uXG4gICAqL1xuICB2aWV3Q2hlY2tIb29rczogSG9va0RhdGF8bnVsbDtcblxuICAvKipcbiAgICogQXJyYXkgb2YgbmdPbkRlc3Ryb3kgaG9va3MgdGhhdCBzaG91bGQgYmUgZXhlY3V0ZWQgd2hlbiB0aGlzIHZpZXcgaXMgZGVzdHJveWVkLlxuICAgKlxuICAgKiBFdmVuIGluZGljZXM6IERpcmVjdGl2ZSBpbmRleFxuICAgKiBPZGQgaW5kaWNlczogSG9vayBmdW5jdGlvblxuICAgKi9cbiAgZGVzdHJveUhvb2tzOiBEZXN0cm95SG9va0RhdGF8bnVsbDtcblxuICAvKipcbiAgICogV2hlbiBhIHZpZXcgaXMgZGVzdHJveWVkLCBsaXN0ZW5lcnMgbmVlZCB0byBiZSByZWxlYXNlZCBhbmQgb3V0cHV0cyBuZWVkIHRvIGJlXG4gICAqIHVuc3Vic2NyaWJlZC4gVGhpcyBjbGVhbnVwIGFycmF5IHN0b3JlcyBib3RoIGxpc3RlbmVyIGRhdGEgKGluIGNodW5rcyBvZiA0KVxuICAgKiBhbmQgb3V0cHV0IGRhdGEgKGluIGNodW5rcyBvZiAyKSBmb3IgYSBwYXJ0aWN1bGFyIHZpZXcuIENvbWJpbmluZyB0aGUgYXJyYXlzXG4gICAqIHNhdmVzIG9uIG1lbW9yeSAoNzAgYnl0ZXMgcGVyIGFycmF5KSBhbmQgb24gYSBmZXcgYnl0ZXMgb2YgY29kZSBzaXplIChmb3IgdHdvXG4gICAqIHNlcGFyYXRlIGZvciBsb29wcykuXG4gICAqXG4gICAqIElmIGl0J3MgYSBuYXRpdmUgRE9NIGxpc3RlbmVyIG9yIG91dHB1dCBzdWJzY3JpcHRpb24gYmVpbmcgc3RvcmVkOlxuICAgKiAxc3QgaW5kZXggaXM6IGV2ZW50IG5hbWUgIGBuYW1lID0gdFZpZXcuY2xlYW51cFtpKzBdYFxuICAgKiAybmQgaW5kZXggaXM6IGluZGV4IG9mIG5hdGl2ZSBlbGVtZW50IG9yIGEgZnVuY3Rpb24gdGhhdCByZXRyaWV2ZXMgZ2xvYmFsIHRhcmdldCAod2luZG93LFxuICAgKiAgICAgICAgICAgICAgIGRvY3VtZW50IG9yIGJvZHkpIHJlZmVyZW5jZSBiYXNlZCBvbiB0aGUgbmF0aXZlIGVsZW1lbnQ6XG4gICAqICAgIGB0eXBlb2YgaWR4T3JUYXJnZXRHZXR0ZXIgPT09ICdmdW5jdGlvbidgOiBnbG9iYWwgdGFyZ2V0IGdldHRlciBmdW5jdGlvblxuICAgKiAgICBgdHlwZW9mIGlkeE9yVGFyZ2V0R2V0dGVyID09PSAnbnVtYmVyJ2A6IGluZGV4IG9mIG5hdGl2ZSBlbGVtZW50XG4gICAqXG4gICAqIDNyZCBpbmRleCBpczogaW5kZXggb2YgbGlzdGVuZXIgZnVuY3Rpb24gYGxpc3RlbmVyID0gbFZpZXdbQ0xFQU5VUF1bdFZpZXcuY2xlYW51cFtpKzJdXWBcbiAgICogNHRoIGluZGV4IGlzOiBgdXNlQ2FwdHVyZU9ySW5keCA9IHRWaWV3LmNsZWFudXBbaSszXWBcbiAgICogICAgYHR5cGVvZiB1c2VDYXB0dXJlT3JJbmR4ID09ICdib29sZWFuJyA6IHVzZUNhcHR1cmUgYm9vbGVhblxuICAgKiAgICBgdHlwZW9mIHVzZUNhcHR1cmVPckluZHggPT0gJ251bWJlcic6XG4gICAqICAgICAgICAgYHVzZUNhcHR1cmVPckluZHggPj0gMGAgYHJlbW92ZUxpc3RlbmVyID0gTFZpZXdbQ0xFQU5VUF1bdXNlQ2FwdHVyZU9ySW5keF1gXG4gICAqICAgICAgICAgYHVzZUNhcHR1cmVPckluZHggPCAgMGAgYHN1YnNjcmlwdGlvbiA9IExWaWV3W0NMRUFOVVBdWy11c2VDYXB0dXJlT3JJbmR4XWBcbiAgICpcbiAgICogSWYgaXQncyBhbiBvdXRwdXQgc3Vic2NyaXB0aW9uIG9yIHF1ZXJ5IGxpc3QgZGVzdHJveSBob29rOlxuICAgKiAxc3QgaW5kZXggaXM6IG91dHB1dCB1bnN1YnNjcmliZSBmdW5jdGlvbiAvIHF1ZXJ5IGxpc3QgZGVzdHJveSBmdW5jdGlvblxuICAgKiAybmQgaW5kZXggaXM6IGluZGV4IG9mIGZ1bmN0aW9uIGNvbnRleHQgaW4gTFZpZXcuY2xlYW51cEluc3RhbmNlc1tdXG4gICAqICAgICAgICAgICAgICAgYHRWaWV3LmNsZWFudXBbaSswXS5jYWxsKGxWaWV3W0NMRUFOVVBdW3RWaWV3LmNsZWFudXBbaSsxXV0pYFxuICAgKi9cbiAgY2xlYW51cDogYW55W118bnVsbDtcblxuICAvKipcbiAgICogQSBsaXN0IG9mIGVsZW1lbnQgaW5kaWNlcyBmb3IgY2hpbGQgY29tcG9uZW50cyB0aGF0IHdpbGwgbmVlZCB0byBiZVxuICAgKiByZWZyZXNoZWQgd2hlbiB0aGUgY3VycmVudCB2aWV3IGhhcyBmaW5pc2hlZCBpdHMgY2hlY2suIFRoZXNlIGluZGljZXMgaGF2ZVxuICAgKiBhbHJlYWR5IGJlZW4gYWRqdXN0ZWQgZm9yIHRoZSBIRUFERVJfT0ZGU0VULlxuICAgKlxuICAgKi9cbiAgY29tcG9uZW50czogbnVtYmVyW118bnVsbDtcblxuICAvKipcbiAgICogQSBjb2xsZWN0aW9uIG9mIHF1ZXJpZXMgdHJhY2tlZCBpbiBhIGdpdmVuIHZpZXcuXG4gICAqL1xuICBxdWVyaWVzOiBUUXVlcmllc3xudWxsO1xuXG4gIC8qKlxuICAgKiBBbiBhcnJheSBvZiBpbmRpY2VzIHBvaW50aW5nIHRvIGRpcmVjdGl2ZXMgd2l0aCBjb250ZW50IHF1ZXJpZXMgYWxvbmdzaWRlIHdpdGggdGhlXG4gICAqIGNvcnJlc3BvbmRpbmcgcXVlcnkgaW5kZXguIEVhY2ggZW50cnkgaW4gdGhpcyBhcnJheSBpcyBhIHR1cGxlIG9mOlxuICAgKiAtIGluZGV4IG9mIHRoZSBmaXJzdCBjb250ZW50IHF1ZXJ5IGluZGV4IGRlY2xhcmVkIGJ5IGEgZ2l2ZW4gZGlyZWN0aXZlO1xuICAgKiAtIGluZGV4IG9mIGEgZGlyZWN0aXZlLlxuICAgKlxuICAgKiBXZSBhcmUgc3RvcmluZyB0aG9zZSBpbmRleGVzIHNvIHdlIGNhbiByZWZyZXNoIGNvbnRlbnQgcXVlcmllcyBhcyBwYXJ0IG9mIGEgdmlldyByZWZyZXNoXG4gICAqIHByb2Nlc3MuXG4gICAqL1xuICBjb250ZW50UXVlcmllczogbnVtYmVyW118bnVsbDtcblxuICAvKipcbiAgICogU2V0IG9mIHNjaGVtYXMgdGhhdCBkZWNsYXJlIGVsZW1lbnRzIHRvIGJlIGFsbG93ZWQgaW5zaWRlIHRoZSB2aWV3LlxuICAgKi9cbiAgc2NoZW1hczogU2NoZW1hTWV0YWRhdGFbXXxudWxsO1xuXG4gIC8qKlxuICAgKiBBcnJheSBvZiBjb25zdGFudHMgZm9yIHRoZSB2aWV3LiBJbmNsdWRlcyBhdHRyaWJ1dGUgYXJyYXlzLCBsb2NhbCBkZWZpbml0aW9uIGFycmF5cyBldGMuXG4gICAqIFVzZWQgZm9yIGRpcmVjdGl2ZSBtYXRjaGluZywgYXR0cmlidXRlIGJpbmRpbmdzLCBsb2NhbCBkZWZpbml0aW9ucyBhbmQgbW9yZS5cbiAgICovXG4gIGNvbnN0czogVENvbnN0YW50c3xudWxsO1xuXG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgdGhhdCB0aGVyZSB3YXMgYW4gZXJyb3IgYmVmb3JlIHdlIG1hbmFnZWQgdG8gY29tcGxldGUgdGhlIGZpcnN0IGNyZWF0ZSBwYXNzIG9mIHRoZVxuICAgKiB2aWV3LiBUaGlzIG1lYW5zIHRoYXQgdGhlIHZpZXcgaXMgbGlrZWx5IGNvcnJ1cHRlZCBhbmQgd2Ugc2hvdWxkIHRyeSB0byByZWNvdmVyIGl0LlxuICAgKi9cbiAgaW5jb21wbGV0ZUZpcnN0UGFzczogYm9vbGVhbjtcbn1cblxuZXhwb3J0IGNvbnN0IGVudW0gUm9vdENvbnRleHRGbGFncyB7XG4gIEVtcHR5ID0gMGIwMCxcbiAgRGV0ZWN0Q2hhbmdlcyA9IDBiMDEsXG4gIEZsdXNoUGxheWVycyA9IDBiMTBcbn1cblxuXG4vKipcbiAqIFJvb3RDb250ZXh0IGNvbnRhaW5zIGluZm9ybWF0aW9uIHdoaWNoIGlzIHNoYXJlZCBmb3IgYWxsIGNvbXBvbmVudHMgd2hpY2hcbiAqIHdlcmUgYm9vdHN0cmFwcGVkIHdpdGgge0BsaW5rIHJlbmRlckNvbXBvbmVudH0uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUm9vdENvbnRleHQge1xuICAvKipcbiAgICogQSBmdW5jdGlvbiB1c2VkIGZvciBzY2hlZHVsaW5nIGNoYW5nZSBkZXRlY3Rpb24gaW4gdGhlIGZ1dHVyZS4gVXN1YWxseVxuICAgKiB0aGlzIGlzIGByZXF1ZXN0QW5pbWF0aW9uRnJhbWVgLlxuICAgKi9cbiAgc2NoZWR1bGVyOiAod29ya0ZuOiAoKSA9PiB2b2lkKSA9PiB2b2lkO1xuXG4gIC8qKlxuICAgKiBBIHByb21pc2Ugd2hpY2ggaXMgcmVzb2x2ZWQgd2hlbiBhbGwgY29tcG9uZW50cyBhcmUgY29uc2lkZXJlZCBjbGVhbiAobm90IGRpcnR5KS5cbiAgICpcbiAgICogVGhpcyBwcm9taXNlIGlzIG92ZXJ3cml0dGVuIGV2ZXJ5IHRpbWUgYSBmaXJzdCBjYWxsIHRvIHtAbGluayBtYXJrRGlydHl9IGlzIGludm9rZWQuXG4gICAqL1xuICBjbGVhbjogUHJvbWlzZTxudWxsPjtcblxuICAvKipcbiAgICogUm9vdENvbXBvbmVudHMgLSBUaGUgY29tcG9uZW50cyB0aGF0IHdlcmUgaW5zdGFudGlhdGVkIGJ5IHRoZSBjYWxsIHRvXG4gICAqIHtAbGluayByZW5kZXJDb21wb25lbnR9LlxuICAgKi9cbiAgY29tcG9uZW50czoge31bXTtcblxuICAvKipcbiAgICogVGhlIHBsYXllciBmbHVzaGluZyBoYW5kbGVyIHRvIGtpY2sgb2ZmIGFsbCBhbmltYXRpb25zXG4gICAqL1xuICBwbGF5ZXJIYW5kbGVyOiBQbGF5ZXJIYW5kbGVyfG51bGw7XG5cbiAgLyoqXG4gICAqIFdoYXQgcmVuZGVyLXJlbGF0ZWQgb3BlcmF0aW9ucyB0byBydW4gb25jZSBhIHNjaGVkdWxlciBoYXMgYmVlbiBzZXRcbiAgICovXG4gIGZsYWdzOiBSb290Q29udGV4dEZsYWdzO1xufVxuXG4vKiogU2luZ2xlIGhvb2sgY2FsbGJhY2sgZnVuY3Rpb24uICovXG5leHBvcnQgdHlwZSBIb29rRm4gPSAoKSA9PiB2b2lkO1xuXG4vKipcbiAqIEluZm9ybWF0aW9uIG5lY2Vzc2FyeSB0byBjYWxsIGEgaG9vay4gRS5nLiB0aGUgY2FsbGJhY2sgdGhhdFxuICogbmVlZHMgdG8gaW52b2tlZCBhbmQgdGhlIGluZGV4IGF0IHdoaWNoIHRvIGZpbmQgaXRzIGNvbnRleHQuXG4gKi9cbmV4cG9ydCB0eXBlIEhvb2tFbnRyeSA9IG51bWJlcnxIb29rRm47XG5cbi8qKlxuICogQXJyYXkgb2YgaG9va3MgdGhhdCBzaG91bGQgYmUgZXhlY3V0ZWQgZm9yIGEgdmlldyBhbmQgdGhlaXIgZGlyZWN0aXZlIGluZGljZXMuXG4gKlxuICogRm9yIGVhY2ggbm9kZSBvZiB0aGUgdmlldywgdGhlIGZvbGxvd2luZyBkYXRhIGlzIHN0b3JlZDpcbiAqIDEpIE5vZGUgaW5kZXggKG9wdGlvbmFsKVxuICogMikgQSBzZXJpZXMgb2YgbnVtYmVyL2Z1bmN0aW9uIHBhaXJzIHdoZXJlOlxuICogIC0gZXZlbiBpbmRpY2VzIGFyZSBkaXJlY3RpdmUgaW5kaWNlc1xuICogIC0gb2RkIGluZGljZXMgYXJlIGhvb2sgZnVuY3Rpb25zXG4gKlxuICogU3BlY2lhbCBjYXNlczpcbiAqICAtIGEgbmVnYXRpdmUgZGlyZWN0aXZlIGluZGV4IGZsYWdzIGFuIGluaXQgaG9vayAobmdPbkluaXQsIG5nQWZ0ZXJDb250ZW50SW5pdCwgbmdBZnRlclZpZXdJbml0KVxuICovXG5leHBvcnQgdHlwZSBIb29rRGF0YSA9IEhvb2tFbnRyeVtdO1xuXG4vKipcbiAqIEFycmF5IG9mIGRlc3Ryb3kgaG9va3MgdGhhdCBzaG91bGQgYmUgZXhlY3V0ZWQgZm9yIGEgdmlldyBhbmQgdGhlaXIgZGlyZWN0aXZlIGluZGljZXMuXG4gKlxuICogVGhlIGFycmF5IGlzIHNldCB1cCBhcyBhIHNlcmllcyBvZiBudW1iZXIvZnVuY3Rpb24gb3IgbnVtYmVyLyhudW1iZXJ8ZnVuY3Rpb24pW106XG4gKiAtIEV2ZW4gaW5kaWNlcyByZXByZXNlbnQgdGhlIGNvbnRleHQgd2l0aCB3aGljaCBob29rcyBzaG91bGQgYmUgY2FsbGVkLlxuICogLSBPZGQgaW5kaWNlcyBhcmUgdGhlIGhvb2sgZnVuY3Rpb25zIHRoZW1zZWx2ZXMuIElmIGEgdmFsdWUgYXQgYW4gb2RkIGluZGV4IGlzIGFuIGFycmF5LFxuICogICBpdCByZXByZXNlbnRzIHRoZSBkZXN0cm95IGhvb2tzIG9mIGEgYG11bHRpYCBwcm92aWRlciB3aGVyZTpcbiAqICAgICAtIEV2ZW4gaW5kaWNlcyByZXByZXNlbnQgdGhlIGluZGV4IG9mIHRoZSBwcm92aWRlciBmb3Igd2hpY2ggd2UndmUgcmVnaXN0ZXJlZCBhIGRlc3Ryb3kgaG9vayxcbiAqICAgICAgIGluc2lkZSBvZiB0aGUgYG11bHRpYCBwcm92aWRlciBhcnJheS5cbiAqICAgICAtIE9kZCBpbmRpY2VzIGFyZSB0aGUgZGVzdHJveSBob29rIGZ1bmN0aW9ucy5cbiAqIEZvciBleGFtcGxlOlxuICogTFZpZXc6IGBbMCwgMSwgMiwgQVNlcnZpY2UsIDQsIFtCU2VydmljZSwgQ1NlcnZpY2UsIERTZXJ2aWNlXV1gXG4gKiBkZXN0cm95SG9va3M6IGBbMywgQVNlcnZpY2UubmdPbkRlc3Ryb3ksIDUsIFswLCBCU2VydmljZS5uZ09uRGVzdHJveSwgMiwgRFNlcnZpY2UubmdPbkRlc3Ryb3ldXWBcbiAqXG4gKiBJbiB0aGUgZXhhbXBsZSBhYm92ZSBgQVNlcnZpY2VgIGlzIGEgdHlwZSBwcm92aWRlciB3aXRoIGFuIGBuZ09uRGVzdHJveWAsIHdoZXJlYXMgYEJTZXJ2aWNlYCxcbiAqIGBDU2VydmljZWAgYW5kIGBEU2VydmljZWAgYXJlIHBhcnQgb2YgYSBgbXVsdGlgIHByb3ZpZGVyIHdoZXJlIG9ubHkgYEJTZXJ2aWNlYCBhbmQgYERTZXJ2aWNlYFxuICogaGF2ZSBhbiBgbmdPbkRlc3Ryb3lgIGhvb2suXG4gKi9cbmV4cG9ydCB0eXBlIERlc3Ryb3lIb29rRGF0YSA9IChIb29rRW50cnl8SG9va0RhdGEpW107XG5cbi8qKlxuICogU3RhdGljIGRhdGEgdGhhdCBjb3JyZXNwb25kcyB0byB0aGUgaW5zdGFuY2Utc3BlY2lmaWMgZGF0YSBhcnJheSBvbiBhbiBMVmlldy5cbiAqXG4gKiBFYWNoIG5vZGUncyBzdGF0aWMgZGF0YSBpcyBzdG9yZWQgaW4gdERhdGEgYXQgdGhlIHNhbWUgaW5kZXggdGhhdCBpdCdzIHN0b3JlZFxuICogaW4gdGhlIGRhdGEgYXJyYXkuICBBbnkgbm9kZXMgdGhhdCBkbyBub3QgaGF2ZSBzdGF0aWMgZGF0YSBzdG9yZSBhIG51bGwgdmFsdWUgaW5cbiAqIHREYXRhIHRvIGF2b2lkIGEgc3BhcnNlIGFycmF5LlxuICpcbiAqIEVhY2ggcGlwZSdzIGRlZmluaXRpb24gaXMgc3RvcmVkIGhlcmUgYXQgdGhlIHNhbWUgaW5kZXggYXMgaXRzIHBpcGUgaW5zdGFuY2UgaW5cbiAqIHRoZSBkYXRhIGFycmF5LlxuICpcbiAqIEVhY2ggaG9zdCBwcm9wZXJ0eSdzIG5hbWUgaXMgc3RvcmVkIGhlcmUgYXQgdGhlIHNhbWUgaW5kZXggYXMgaXRzIHZhbHVlIGluIHRoZVxuICogZGF0YSBhcnJheS5cbiAqXG4gKiBFYWNoIHByb3BlcnR5IGJpbmRpbmcgbmFtZSBpcyBzdG9yZWQgaGVyZSBhdCB0aGUgc2FtZSBpbmRleCBhcyBpdHMgdmFsdWUgaW5cbiAqIHRoZSBkYXRhIGFycmF5LiBJZiB0aGUgYmluZGluZyBpcyBhbiBpbnRlcnBvbGF0aW9uLCB0aGUgc3RhdGljIHN0cmluZyB2YWx1ZXNcbiAqIGFyZSBzdG9yZWQgcGFyYWxsZWwgdG8gdGhlIGR5bmFtaWMgdmFsdWVzLiBFeGFtcGxlOlxuICpcbiAqIGlkPVwicHJlZml4IHt7IHYwIH19IGEge3sgdjEgfX0gYiB7eyB2MiB9fSBzdWZmaXhcIlxuICpcbiAqIExWaWV3ICAgICAgIHwgICBUVmlldy5kYXRhXG4gKi0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuICogIHYwIHZhbHVlICAgfCAgICdhJ1xuICogIHYxIHZhbHVlICAgfCAgICdiJ1xuICogIHYyIHZhbHVlICAgfCAgIGlkIO+/vSBwcmVmaXgg77+9IHN1ZmZpeFxuICpcbiAqIEluamVjdG9yIGJsb29tIGZpbHRlcnMgYXJlIGFsc28gc3RvcmVkIGhlcmUuXG4gKi9cbmV4cG9ydCB0eXBlIFREYXRhID0gKFROb2RlfFBpcGVEZWY8YW55PnxEaXJlY3RpdmVEZWY8YW55PnxDb21wb25lbnREZWY8YW55PnxudW1iZXJ8VFN0eWxpbmdSYW5nZXxcbiAgICAgICAgICAgICAgICAgICAgIFRTdHlsaW5nS2V5fFByb3ZpZGVyVG9rZW48YW55PnxUSTE4bnxJMThuVXBkYXRlT3BDb2Rlc3xUSWN1fG51bGx8c3RyaW5nKVtdO1xuXG4vLyBOb3RlOiBUaGlzIGhhY2sgaXMgbmVjZXNzYXJ5IHNvIHdlIGRvbid0IGVycm9uZW91c2x5IGdldCBhIGNpcmN1bGFyIGRlcGVuZGVuY3lcbi8vIGZhaWx1cmUgYmFzZWQgb24gdHlwZXMuXG5leHBvcnQgY29uc3QgdW51c2VkVmFsdWVFeHBvcnRUb1BsYWNhdGVBamQgPSAxO1xuXG4vKipcbiAqIEh1bWFuIHJlYWRhYmxlIHZlcnNpb24gb2YgdGhlIGBMVmlld2AuXG4gKlxuICogYExWaWV3YCBpcyBhIGRhdGEgc3RydWN0dXJlIHVzZWQgaW50ZXJuYWxseSB0byBrZWVwIHRyYWNrIG9mIHZpZXdzLiBUaGUgYExWaWV3YCBpcyBkZXNpZ25lZCBmb3JcbiAqIGVmZmljaWVuY3kgYW5kIHNvIGF0IHRpbWVzIGl0IGlzIGRpZmZpY3VsdCB0byByZWFkIG9yIHdyaXRlIHRlc3RzIHdoaWNoIGFzc2VydCBvbiBpdHMgdmFsdWVzLiBGb3JcbiAqIHRoaXMgcmVhc29uIHdoZW4gYG5nRGV2TW9kZWAgaXMgdHJ1ZSB3ZSBwYXRjaCBhIGBMVmlldy5kZWJ1Z2AgcHJvcGVydHkgd2hpY2ggcG9pbnRzIHRvXG4gKiBgTFZpZXdEZWJ1Z2AgZm9yIGVhc2llciBkZWJ1Z2dpbmcgYW5kIHRlc3Qgd3JpdGluZy4gSXQgaXMgdGhlIGludGVudCBvZiBgTFZpZXdEZWJ1Z2AgdG8gYmUgdXNlZFxuICogaW4gdGVzdHMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTFZpZXdEZWJ1ZyB7XG4gIC8qKlxuICAgKiBGbGFncyBhc3NvY2lhdGVkIHdpdGggdGhlIGBMVmlld2AgdW5wYWNrZWQgaW50byBhIG1vcmUgcmVhZGFibGUgc3RhdGUuXG4gICAqXG4gICAqIFNlZSBgTFZpZXdGbGFnc2AgZm9yIHRoZSBmbGFnIG1lYW5pbmdzLlxuICAgKi9cbiAgcmVhZG9ubHkgZmxhZ3M6IHtcbiAgICBpbml0UGhhc2VTdGF0ZTogbnVtYmVyLFxuICAgIGNyZWF0aW9uTW9kZTogYm9vbGVhbixcbiAgICBmaXJzdFZpZXdQYXNzOiBib29sZWFuLFxuICAgIGNoZWNrQWx3YXlzOiBib29sZWFuLFxuICAgIGRpcnR5OiBib29sZWFuLFxuICAgIGF0dGFjaGVkOiBib29sZWFuLFxuICAgIGRlc3Ryb3llZDogYm9vbGVhbixcbiAgICBpc1Jvb3Q6IGJvb2xlYW4sXG4gICAgaW5kZXhXaXRoaW5Jbml0UGhhc2U6IG51bWJlcixcbiAgfTtcblxuICAvKipcbiAgICogQXNzb2NpYXRlZCBUVmlld1xuICAgKi9cbiAgcmVhZG9ubHkgdFZpZXc6IFRWaWV3O1xuXG4gIC8qKlxuICAgKiBQYXJlbnQgdmlldyAob3IgY29udGFpbmVyKVxuICAgKi9cbiAgcmVhZG9ubHkgcGFyZW50OiBMVmlld0RlYnVnfExDb250YWluZXJEZWJ1Z3xudWxsO1xuXG4gIC8qKlxuICAgKiBOZXh0IHNpYmxpbmcgdG8gdGhlIGBMVmlld2AuXG4gICAqL1xuICByZWFkb25seSBuZXh0OiBMVmlld0RlYnVnfExDb250YWluZXJEZWJ1Z3xudWxsO1xuXG4gIC8qKlxuICAgKiBUaGUgY29udGV4dCB1c2VkIGZvciBldmFsdWF0aW9uIG9mIHRoZSBgTFZpZXdgXG4gICAqXG4gICAqIChVc3VhbGx5IHRoZSBjb21wb25lbnQpXG4gICAqL1xuICByZWFkb25seSBjb250ZXh0OiB7fXxudWxsO1xuXG4gIC8qKlxuICAgKiBIaWVyYXJjaGljYWwgdHJlZSBvZiBub2Rlcy5cbiAgICovXG4gIHJlYWRvbmx5IG5vZGVzOiBEZWJ1Z05vZGVbXTtcblxuICAvKipcbiAgICogVGVtcGxhdGUgc3RydWN0dXJlIChubyBpbnN0YW5jZSBkYXRhKS5cbiAgICogKFNob3dzIGhvdyBUTm9kZXMgYXJlIGNvbm5lY3RlZClcbiAgICovXG4gIHJlYWRvbmx5IHRlbXBsYXRlOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEhUTUwgcmVwcmVzZW50YXRpb24gb2YgdGhlIGBMVmlld2AuXG4gICAqXG4gICAqIFRoaXMgaXMgb25seSBhcHByb3hpbWF0ZSB0byBhY3R1YWwgSFRNTCBhcyBjaGlsZCBgTFZpZXdgcyBhcmUgcmVtb3ZlZC5cbiAgICovXG4gIHJlYWRvbmx5IGh0bWw6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGhvc3QgZWxlbWVudCB0byB3aGljaCB0aGlzIGBMVmlld2AgaXMgYXR0YWNoZWQuXG4gICAqL1xuICByZWFkb25seSBob3N0SFRNTDogc3RyaW5nfG51bGw7XG5cbiAgLyoqXG4gICAqIENoaWxkIGBMVmlld2BzXG4gICAqL1xuICByZWFkb25seSBjaGlsZFZpZXdzOiBBcnJheTxMVmlld0RlYnVnfExDb250YWluZXJEZWJ1Zz47XG5cbiAgLyoqXG4gICAqIFN1YiByYW5nZSBvZiBgTFZpZXdgIGNvbnRhaW5pbmcgZGVjbHMgKERPTSBlbGVtZW50cykuXG4gICAqL1xuICByZWFkb25seSBkZWNsczogTFZpZXdEZWJ1Z1JhbmdlO1xuXG4gIC8qKlxuICAgKiBTdWIgcmFuZ2Ugb2YgYExWaWV3YCBjb250YWluaW5nIHZhcnMgKGJpbmRpbmdzKS5cbiAgICovXG4gIHJlYWRvbmx5IHZhcnM6IExWaWV3RGVidWdSYW5nZTtcblxuICAvKipcbiAgICogU3ViIHJhbmdlIG9mIGBMVmlld2AgY29udGFpbmluZyBleHBhbmRvICh1c2VkIGJ5IERJKS5cbiAgICovXG4gIHJlYWRvbmx5IGV4cGFuZG86IExWaWV3RGVidWdSYW5nZTtcbn1cblxuLyoqXG4gKiBIdW1hbiByZWFkYWJsZSB2ZXJzaW9uIG9mIHRoZSBgTENvbnRhaW5lcmBcbiAqXG4gKiBgTENvbnRhaW5lcmAgaXMgYSBkYXRhIHN0cnVjdHVyZSB1c2VkIGludGVybmFsbHkgdG8ga2VlcCB0cmFjayBvZiBjaGlsZCB2aWV3cy4gVGhlIGBMQ29udGFpbmVyYFxuICogaXMgZGVzaWduZWQgZm9yIGVmZmljaWVuY3kgYW5kIHNvIGF0IHRpbWVzIGl0IGlzIGRpZmZpY3VsdCB0byByZWFkIG9yIHdyaXRlIHRlc3RzIHdoaWNoIGFzc2VydCBvblxuICogaXRzIHZhbHVlcy4gRm9yIHRoaXMgcmVhc29uIHdoZW4gYG5nRGV2TW9kZWAgaXMgdHJ1ZSB3ZSBwYXRjaCBhIGBMQ29udGFpbmVyLmRlYnVnYCBwcm9wZXJ0eSB3aGljaFxuICogcG9pbnRzIHRvIGBMQ29udGFpbmVyRGVidWdgIGZvciBlYXNpZXIgZGVidWdnaW5nIGFuZCB0ZXN0IHdyaXRpbmcuIEl0IGlzIHRoZSBpbnRlbnQgb2ZcbiAqIGBMQ29udGFpbmVyRGVidWdgIHRvIGJlIHVzZWQgaW4gdGVzdHMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTENvbnRhaW5lckRlYnVnIHtcbiAgcmVhZG9ubHkgbmF0aXZlOiBSQ29tbWVudDtcbiAgLyoqXG4gICAqIENoaWxkIGBMVmlld2BzLlxuICAgKi9cbiAgcmVhZG9ubHkgdmlld3M6IExWaWV3RGVidWdbXTtcbiAgcmVhZG9ubHkgcGFyZW50OiBMVmlld0RlYnVnfG51bGw7XG4gIHJlYWRvbmx5IG1vdmVkVmlld3M6IExWaWV3W118bnVsbDtcbiAgcmVhZG9ubHkgaG9zdDogUkVsZW1lbnR8UkNvbW1lbnR8TFZpZXc7XG4gIHJlYWRvbmx5IG5leHQ6IExWaWV3RGVidWd8TENvbnRhaW5lckRlYnVnfG51bGw7XG4gIHJlYWRvbmx5IGhhc1RyYW5zcGxhbnRlZFZpZXdzOiBib29sZWFuO1xufVxuXG5cblxuLyoqXG4gKiBgTFZpZXdgIGlzIHN1YmRpdmlkZWQgdG8gcmFuZ2VzIHdoZXJlIHRoZSBhY3R1YWwgZGF0YSBpcyBzdG9yZWQuIFNvbWUgb2YgdGhlc2UgcmFuZ2VzIHN1Y2ggYXNcbiAqIGBkZWNsc2AgYW5kIGB2YXJzYCBhcmUga25vd24gYXQgY29tcGlsZSB0aW1lLiBPdGhlciBzdWNoIGFzIGBpMThuYCBhbmQgYGV4cGFuZG9gIGFyZSBydW50aW1lIG9ubHlcbiAqIGNvbmNlcHRzLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIExWaWV3RGVidWdSYW5nZSB7XG4gIC8qKlxuICAgKiBUaGUgc3RhcnRpbmcgaW5kZXggaW4gYExWaWV3YCB3aGVyZSB0aGUgcmFuZ2UgYmVnaW5zLiAoSW5jbHVzaXZlKVxuICAgKi9cbiAgc3RhcnQ6IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIGVuZGluZyBpbmRleCBpbiBgTFZpZXdgIHdoZXJlIHRoZSByYW5nZSBlbmRzLiAoRXhjbHVzaXZlKVxuICAgKi9cbiAgZW5kOiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBsZW5ndGggb2YgdGhlIHJhbmdlXG4gICAqL1xuICBsZW5ndGg6IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIG1lcmdlZCBjb250ZW50IG9mIHRoZSByYW5nZS4gYHRgIGNvbnRhaW5zIGRhdGEgZnJvbSBgVFZpZXcuZGF0YWAgYW5kIGBsYCBjb250YWlucyBgTFZpZXdgXG4gICAqIGRhdGEgYXQgYW4gaW5kZXguXG4gICAqL1xuICBjb250ZW50OiBMVmlld0RlYnVnUmFuZ2VDb250ZW50W107XG59XG5cbi8qKlxuICogRm9yIGNvbnZlbmllbmNlIHRoZSBzdGF0aWMgYW5kIGluc3RhbmNlIHBvcnRpb25zIG9mIGBUVmlld2AgYW5kIGBMVmlld2AgYXJlIG1lcmdlZCBpbnRvIGEgc2luZ2xlXG4gKiBvYmplY3QgaW4gYExWaWV3UmFuZ2VgLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIExWaWV3RGVidWdSYW5nZUNvbnRlbnQge1xuICAvKipcbiAgICogSW5kZXggaW50byBvcmlnaW5hbCBgTFZpZXdgIG9yIGBUVmlldy5kYXRhYC5cbiAgICovXG4gIGluZGV4OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFZhbHVlIGZyb20gdGhlIGBUVmlldy5kYXRhW2luZGV4XWAgbG9jYXRpb24uXG4gICAqL1xuICB0OiBhbnk7XG5cbiAgLyoqXG4gICAqIFZhbHVlIGZyb20gdGhlIGBMVmlld1tpbmRleF1gIGxvY2F0aW9uLlxuICAgKi9cbiAgbDogYW55O1xufVxuXG5cbi8qKlxuICogQSBsb2dpY2FsIG5vZGUgd2hpY2ggY29tcHJpc2UgaW50byBgTFZpZXdgcy5cbiAqXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRGVidWdOb2RlIHtcbiAgLyoqXG4gICAqIEhUTUwgcmVwcmVzZW50YXRpb24gb2YgdGhlIG5vZGUuXG4gICAqL1xuICBodG1sOiBzdHJpbmd8bnVsbDtcblxuICAvKipcbiAgICogQXNzb2NpYXRlZCBgVE5vZGVgXG4gICAqL1xuICB0Tm9kZTogVE5vZGU7XG5cbiAgLyoqXG4gICAqIEh1bWFuIHJlYWRhYmxlIG5vZGUgdHlwZS5cbiAgICovXG4gIHR5cGU6IHN0cmluZztcblxuICAvKipcbiAgICogRE9NIG5hdGl2ZSBub2RlLlxuICAgKi9cbiAgbmF0aXZlOiBOb2RlO1xuXG4gIC8qKlxuICAgKiBDaGlsZCBub2Rlc1xuICAgKi9cbiAgY2hpbGRyZW46IERlYnVnTm9kZVtdO1xuXG4gIC8qKlxuICAgKiBBIGxpc3Qgb2YgQ29tcG9uZW50L0RpcmVjdGl2ZSB0eXBlcyB3aGljaCBuZWVkIHRvIGJlIGluc3RhbnRpYXRlZCBhbiB0aGlzIGxvY2F0aW9uLlxuICAgKi9cbiAgZmFjdG9yaWVzOiBUeXBlPHVua25vd24+W107XG5cbiAgLyoqXG4gICAqIEEgbGlzdCBvZiBDb21wb25lbnQvRGlyZWN0aXZlIGluc3RhbmNlcyB3aGljaCB3ZXJlIGluc3RhbnRpYXRlZCBhbiB0aGlzIGxvY2F0aW9uLlxuICAgKi9cbiAgaW5zdGFuY2VzOiB1bmtub3duW107XG5cbiAgLyoqXG4gICAqIE5vZGVJbmplY3RvciBpbmZvcm1hdGlvbi5cbiAgICovXG4gIGluamVjdG9yOiBOb2RlSW5qZWN0b3JEZWJ1ZztcblxuICAvKipcbiAgICogSW5qZWN0b3IgcmVzb2x1dGlvbiBwYXRoLlxuICAgKi9cbiAgaW5qZWN0b3JSZXNvbHV0aW9uUGF0aDogYW55O1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIE5vZGVJbmplY3RvckRlYnVnIHtcbiAgLyoqXG4gICAqIEluc3RhbmNlIGJsb29tLiBEb2VzIHRoZSBjdXJyZW50IGluamVjdG9yIGhhdmUgYSBwcm92aWRlciB3aXRoIGEgZ2l2ZW4gYmxvb20gbWFzay5cbiAgICovXG4gIGJsb29tOiBzdHJpbmc7XG5cblxuICAvKipcbiAgICogQ3VtdWxhdGl2ZSBibG9vbS4gRG8gYW55IG9mIHRoZSBhYm92ZSBpbmplY3RvcnMgaGF2ZSBhIHByb3ZpZGVyIHdpdGggYSBnaXZlbiBibG9vbSBtYXNrLlxuICAgKi9cbiAgY3VtdWxhdGl2ZUJsb29tOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEEgbGlzdCBvZiBwcm92aWRlcnMgYXNzb2NpYXRlZCB3aXRoIHRoaXMgaW5qZWN0b3IuXG4gICAqL1xuICBwcm92aWRlcnM6IChUeXBlPHVua25vd24+fERpcmVjdGl2ZURlZjx1bmtub3duPnxDb21wb25lbnREZWY8dW5rbm93bj4pW107XG5cbiAgLyoqXG4gICAqIEEgbGlzdCBvZiBwcm92aWRlcnMgYXNzb2NpYXRlZCB3aXRoIHRoaXMgaW5qZWN0b3IgdmlzaWJsZSB0byB0aGUgdmlldyBvZiB0aGUgY29tcG9uZW50IG9ubHkuXG4gICAqL1xuICB2aWV3UHJvdmlkZXJzOiBUeXBlPHVua25vd24+W107XG5cblxuICAvKipcbiAgICogTG9jYXRpb24gb2YgdGhlIHBhcmVudCBgVE5vZGVgLlxuICAgKi9cbiAgcGFyZW50SW5qZWN0b3JJbmRleDogbnVtYmVyO1xufVxuIl19