/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ERROR_DETAILS_PAGE_BASE_URL } from './error_details_base_url';
/**
 * Class that represents a runtime error.
 * Formats and outputs the error message in a consistent way.
 *
 * Example:
 * ```
 *  throw new RuntimeError(
 *    RuntimeErrorCode.INJECTOR_ALREADY_DESTROYED,
 *    ngDevMode && 'Injector has already been destroyed.');
 * ```
 *
 * Note: the `message` argument contains a descriptive error message as a string in development
 * mode (when the `ngDevMode` is defined). In production mode (after tree-shaking pass), the
 * `message` argument becomes `false`, thus we account for it in the typings and the runtime logic.
 */
export class RuntimeError extends Error {
    constructor(code, message) {
        super(formatRuntimeError(code, message));
        this.code = code;
    }
}
/**
 * Called to format a runtime error.
 * See additional info on the `message` argument type in the `RuntimeError` class description.
 */
export function formatRuntimeError(code, message) {
    // Error code might be a negative number, which is a special marker that instructs the logic to
    // generate a link to the error details page on angular.io.
    const fullCode = `NG0${Math.abs(code)}`;
    let errorMessage = `${fullCode}${message ? ': ' + message : ''}`;
    if (ngDevMode && code < 0) {
        errorMessage = `${errorMessage}. Find more at ${ERROR_DETAILS_PAGE_BASE_URL}/${fullCode}`;
    }
    return errorMessage;
}
//# sourceMappingURL=data:application/json;base64,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