/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ɵRuntimeError as RuntimeError } from '@angular/core';
const NG_DEV_MODE = typeof ngDevMode === 'undefined' || !!ngDevMode;
const LINE_START = '\n - ';
export function invalidTimingValue(exp) {
    return new RuntimeError(3000 /* INVALID_TIMING_VALUE */, NG_DEV_MODE && `The provided timing value "${exp}" is invalid.`);
}
export function negativeStepValue() {
    return new RuntimeError(3100 /* NEGATIVE_STEP_VALUE */, NG_DEV_MODE && 'Duration values below 0 are not allowed for this animation step.');
}
export function negativeDelayValue() {
    return new RuntimeError(3101 /* NEGATIVE_DELAY_VALUE */, NG_DEV_MODE && 'Delay values below 0 are not allowed for this animation step.');
}
export function invalidStyleParams(varName) {
    return new RuntimeError(3001 /* INVALID_STYLE_PARAMS */, NG_DEV_MODE &&
        `Unable to resolve the local animation param ${varName} in the given list of values`);
}
export function invalidParamValue(varName) {
    return new RuntimeError(3003 /* INVALID_PARAM_VALUE */, NG_DEV_MODE && `Please provide a value for the animation param ${varName}`);
}
export function invalidNodeType(nodeType) {
    return new RuntimeError(3004 /* INVALID_NODE_TYPE */, NG_DEV_MODE && `Unable to resolve animation metadata node #${nodeType}`);
}
export function invalidCssUnitValue(userProvidedProperty, value) {
    return new RuntimeError(3005 /* INVALID_CSS_UNIT_VALUE */, NG_DEV_MODE && `Please provide a CSS unit value for ${userProvidedProperty}:${value}`);
}
export function invalidTrigger() {
    return new RuntimeError(3006 /* INVALID_TRIGGER */, NG_DEV_MODE &&
        'animation triggers cannot be prefixed with an `@` sign (e.g. trigger(\'@foo\', [...]))');
}
export function invalidDefinition() {
    return new RuntimeError(3007 /* INVALID_DEFINITION */, NG_DEV_MODE && 'only state() and transition() definitions can sit inside of a trigger()');
}
export function invalidState(metadataName, missingSubs) {
    return new RuntimeError(3008 /* INVALID_STATE */, NG_DEV_MODE &&
        `state("${metadataName}", ...) must define default values for all the following style substitutions: ${missingSubs.join(', ')}`);
}
export function invalidStyleValue(value) {
    return new RuntimeError(3002 /* INVALID_STYLE_VALUE */, NG_DEV_MODE && `The provided style string value ${value} is not allowed.`);
}
export function invalidProperty(prop) {
    return new RuntimeError(3009 /* INVALID_PROPERTY */, NG_DEV_MODE &&
        `The provided animation property "${prop}" is not a supported CSS property for animations`);
}
export function invalidParallelAnimation(prop, firstStart, firstEnd, secondStart, secondEnd) {
    return new RuntimeError(3010 /* INVALID_PARALLEL_ANIMATION */, NG_DEV_MODE &&
        `The CSS property "${prop}" that exists between the times of "${firstStart}ms" and "${firstEnd}ms" is also being animated in a parallel animation between the times of "${secondStart}ms" and "${secondEnd}ms"`);
}
export function invalidKeyframes() {
    return new RuntimeError(3011 /* INVALID_KEYFRAMES */, NG_DEV_MODE && `keyframes() must be placed inside of a call to animate()`);
}
export function invalidOffset() {
    return new RuntimeError(3012 /* INVALID_OFFSET */, NG_DEV_MODE && `Please ensure that all keyframe offsets are between 0 and 1`);
}
export function keyframeOffsetsOutOfOrder() {
    return new RuntimeError(3200 /* KEYFRAME_OFFSETS_OUT_OF_ORDER */, NG_DEV_MODE && `Please ensure that all keyframe offsets are in order`);
}
export function keyframesMissingOffsets() {
    return new RuntimeError(3202 /* KEYFRAMES_MISSING_OFFSETS */, NG_DEV_MODE && `Not all style() steps within the declared keyframes() contain offsets`);
}
export function invalidStagger() {
    return new RuntimeError(3013 /* INVALID_STAGGER */, NG_DEV_MODE && `stagger() can only be used inside of query()`);
}
export function invalidQuery(selector) {
    return new RuntimeError(3014 /* INVALID_QUERY */, NG_DEV_MODE &&
        `\`query("${selector}")\` returned zero elements. (Use \`query("${selector}", { optional: true })\` if you wish to allow this.)`);
}
export function invalidExpression(expr) {
    return new RuntimeError(3015 /* INVALID_EXPRESSION */, NG_DEV_MODE && `The provided transition expression "${expr}" is not supported`);
}
export function invalidTransitionAlias(alias) {
    return new RuntimeError(3016 /* INVALID_TRANSITION_ALIAS */, NG_DEV_MODE && `The transition alias value "${alias}" is not supported`);
}
export function validationFailed(errors) {
    return new RuntimeError(3500 /* VALIDATION_FAILED */, NG_DEV_MODE && `animation validation failed:\n${errors.map(err => err.message).join('\n')}`);
}
export function buildingFailed(errors) {
    return new RuntimeError(3501 /* BUILDING_FAILED */, NG_DEV_MODE && `animation building failed:\n${errors.map(err => err.message).join('\n')}`);
}
export function triggerBuildFailed(name, errors) {
    return new RuntimeError(3404 /* TRIGGER_BUILD_FAILED */, NG_DEV_MODE &&
        `The animation trigger "${name}" has failed to build due to the following errors:\n - ${errors.map(err => err.message).join('\n - ')}`);
}
export function animationFailed(errors) {
    return new RuntimeError(3502 /* ANIMATION_FAILED */, NG_DEV_MODE &&
        `Unable to animate due to the following errors:${LINE_START}${errors.map(err => err.message).join(LINE_START)}`);
}
export function registerFailed(errors) {
    return new RuntimeError(3503 /* REGISTRATION_FAILED */, NG_DEV_MODE &&
        `Unable to build the animation due to the following errors: ${errors.map(err => err.message).join('\n')}`);
}
export function missingOrDestroyedAnimation() {
    return new RuntimeError(3300 /* MISSING_OR_DESTROYED_ANIMATION */, NG_DEV_MODE && 'The requested animation doesn\'t exist or has already been destroyed');
}
export function createAnimationFailed(errors) {
    return new RuntimeError(3504 /* CREATE_ANIMATION_FAILED */, NG_DEV_MODE &&
        `Unable to create the animation due to the following errors:${errors.map(err => err.message).join('\n')}`);
}
export function missingPlayer(id) {
    return new RuntimeError(3301 /* MISSING_PLAYER */, NG_DEV_MODE && `Unable to find the timeline player referenced by ${id}`);
}
export function missingTrigger(phase, name) {
    return new RuntimeError(3302 /* MISSING_TRIGGER */, NG_DEV_MODE &&
        `Unable to listen on the animation trigger event "${phase}" because the animation trigger "${name}" doesn\'t exist!`);
}
export function missingEvent(name) {
    return new RuntimeError(3303 /* MISSING_EVENT */, NG_DEV_MODE &&
        `Unable to listen on the animation trigger "${name}" because the provided event is undefined!`);
}
export function unsupportedTriggerEvent(phase, name) {
    return new RuntimeError(3400 /* UNSUPPORTED_TRIGGER_EVENT */, NG_DEV_MODE &&
        `The provided animation trigger event "${phase}" for the animation trigger "${name}" is not supported!`);
}
export function unregisteredTrigger(name) {
    return new RuntimeError(3401 /* UNREGISTERED_TRIGGER */, NG_DEV_MODE && `The provided animation trigger "${name}" has not been registered!`);
}
export function triggerTransitionsFailed(errors) {
    return new RuntimeError(3402 /* TRIGGER_TRANSITIONS_FAILED */, NG_DEV_MODE &&
        `Unable to process animations due to the following failed trigger transitions\n ${errors.map(err => err.message).join('\n')}`);
}
export function triggerParsingFailed(name, errors) {
    return new RuntimeError(3403 /* TRIGGER_PARSING_FAILED */, NG_DEV_MODE &&
        `Animation parsing for the ${name} trigger have failed:${LINE_START}${errors.map(err => err.message).join(LINE_START)}`);
}
export function transitionFailed(name, errors) {
    return new RuntimeError(3505 /* TRANSITION_FAILED */, NG_DEV_MODE &&
        `@${name} has failed due to:\n ${errors.map(err => err.message).join('\n- ')}`);
}
//# sourceMappingURL=data:application/json;base64,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